<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author    SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZend_Validate
 * @version    $Id: Abstract.php 8113 2008-02-18 13:15:27Z matthew $
 * @category   SZend
 */

/**
 * @see SZendValidateInterface
 */
require_once dirname(__FILE__).'/Interface.php';

abstract class SZendValidateAbstract implements SZendValidateInterface
{
    /**
     * Default translation object for all validate objects
     * @var SZend_Translate
     */
    protected static $defaultTranslator;
    /**
     * The value to be validated
     *
     * @var mixed
     */
    protected $value;
    /**
     * Additional variables available for validation failure messages
     *
     * @var array
     */
    protected $messageVariables = array();
    /**
     * Validation failure message template definitions
     *
     * @var array
     */
    protected $messageTemplates = array();
    /**
     * Array of validation failure messages
     *
     * @var array
     */
    protected $messages = array();
    /**
     * Flag indidcating whether or not value should be obfuscated in error
     * messages
     * @var bool
     */
    protected $obscureValue = false;
    /**
     * Array of validation failure message codes
     *
     * @var array
     * @deprecated Since 1.5.0
     */
    protected $errors = array();
    /**
     * Translation object
     * @var SZend_Translate
     */
    protected $translator;

    /**
     * Returns array of validation failure messages
     *
     * @return array
     */
    public function getMessages()
    {
        return $this->messages;
    }

    /**
     * Sets validation failure message templates given as an array, where the array keys are the message keys,
     * and the array values are the message template strings.
     *
     * @param array $messages
     * @return SZendValidateAbstract
     */
    public function setMessages(array $messages)
    {
        foreach ($messages as $key => $message) {
            $this->setMessage($message, $key);
        }

        return $this;
    }

    /**
     * Sets the validation failure message template for a particular key
     *
     * @param string $messageString
     * @param string $messageKey OPTIONAL
     * @return SZendValidateAbstract Provides a fluent interface
     * @throws SZend_Validate_Exception
     */
    public function setMessage($messageString, $messageKey = null)
    {
        if ($messageKey === null) {
            $keys = array_keys($this->messageTemplates);
            $messageKey = current($keys);
        }
        if (!array_key_exists($messageKey, $this->messageTemplates)) {
            require_once 'SZend/Validate/Exception.php';
            throw new SZend_Validate_Exception("No message template exists for key '$messageKey'");
        }
        $this->messageTemplates[$messageKey] = $messageString;

        return $this;
    }

    /**
     * Returns an array of the names of variables that are used in constructing validation failure messages
     *
     * @return array
     */
    public function getMessageVariables()
    {
        return array_keys($this->messageVariables);
    }

    /**
     * Magic function returns the value of the requested property, if and only if it is the value or a
     * message variable.
     *
     * @param string $property
     * @return mixed
     * @throws SZend_Validate_Exception
     */
    public function __get($property)
    {
        if ($property == 'value') {
            return $this->value;
        }
        if (array_key_exists($property, $this->messageVariables)) {
            return $this->{$this->messageVariables[$property]};
        }
        /**
         * @see SZend_Validate_Exception
         */
        require_once 'SZend/Validate/Exception.php';
        throw new SZend_Validate_Exception("No property exists by the name '$property'");
    }

    /**
     * Returns array of validation failure message codes
     *
     * @return array
     * @deprecated Since 1.5.0
     */
    public function getErrors()
    {
        return $this->_errors;
    }

    /**
     * @param string $messageKey OPTIONAL
     * @param string $value OPTIONAL
     * @return void
     */
    protected function error($messageKey = null, $value = null)
    {
        if ($messageKey === null) {
            $keys = array_keys($this->messageTemplates);
            $messageKey = current($keys);
        }
        if ($value === null) {
            $value = $this->value;
        }
        $this->_errors[] = $messageKey;
        $this->messages[$messageKey] = $this->createMessage($messageKey, $value);
    }

    /**
     * Constructs and returns a validation failure message with the given message key and value.
     *
     * Returns null if and only if $messageKey does not correspond to an existing template.
     *
     * If a translator is available and a translation exists for $messageKey,
     * the translation will be used.
     *
     * @param string $messageKey
     * @param string $value
     * @return string
     */
    protected function createMessage($messageKey, $value)
    {
        if (!array_key_exists($messageKey, $this->messageTemplates)) {
            return null;
        }

        $message = $this->messageTemplates[$messageKey];

        if (null !== ($translator = $this->getTranslator())) {
            if ($translator->isTranslated($messageKey)) {
                $message = $translator->translate($messageKey);
            }
        }

        if ($this->getObscureValue()) {
            $value = str_repeat('*', Tools::strlen($value));
        }

        $message = str_replace('%value%', (string)$value, $message);
        foreach ($this->messageVariables as $ident => $property) {
            $message = str_replace("%$ident%", $this->$property, $message);
        }

        return $message;
    }

    /**
     * Return translation object
     *
     * @return SZend_Translate_Adapter|null
     */
    public function getTranslator()
    {
        if (null === $this->translator) {
            return self::getDefaultTranslator();
        }

        return $this->translator;
    }

    /**
     * Set translation object
     *
     * @param SZend_Translate|SZend_Translate_Adapter|null $translator
     * @return SZendValidateAbstract
     */
    public function setTranslator($translator = null)
    {
        if ((null === $translator) || ($translator instanceof SZend_Translate_Adapter)) {
            $this->translator = $translator;
        } elseif ($translator instanceof SZend_Translate) {
            $this->translator = $translator->getAdapter();
        } else {
            require_once 'SZend/Validate/Exception.php';
            throw new SZend_Validate_Exception('Invalid translator specified');
        }

        return $this;
    }

    /**
     * Get default translation object for all validate objects
     *
     * @return SZend_Translate_Adapter|null
     */
    public static function getDefaultTranslator()
    {
        if (null === self::$defaultTranslator) {
        }

        return self::$defaultTranslator;
    }

    /**
     * Set default translation object for all validate objects
     *
     * @param SZend_Translate|SZend_Translate_Adapter|null $translator
     * @return void
     */
    public static function setDefaultTranslator($translator = null)
    {
        if ((null === $translator) || ($translator instanceof SZend_Translate_Adapter)) {
            self::$defaultTranslator = $translator;
        } elseif ($translator instanceof SZend_Translate) {
            self::$defaultTranslator = $translator->getAdapter();
        } else {
            require_once 'SZend/Validate/Exception.php';
            throw new SZend_Validate_Exception('Invalid translator specified');
        }
    }

    /**
     * Retrieve flag indicating whether or not value should be obfuscated in
     * messages
     *
     * @return bool
     */
    public function getObscureValue()
    {
        return $this->obscureValue;
    }

    /**
     * Set flag indicating whether or not value should be obfuscated in messages
     *
     * @param bool $flag
     * @return SZendValidateAbstract
     */
    public function setObscureValue($flag)
    {
        $this->obscureValue = (bool)$flag;

        return $this;
    }

    /**
     * Sets the value to be validated and clears the messages and errors arrays
     *
     * @param mixed $value
     * @return void
     */
    protected function setValue($value)
    {
        $this->value = $value;
        $this->messages = array();
        $this->_errors = array();
    }
}
