<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author    SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZendJson
 * @category   SZend
 */

/**
 * SZendJsonException
 */
require_once dirname(__FILE__).'/../Exception.php';

/**
 * Encode PHP constructs to JSON
 */
class SZendJsonEncoder
{
    /**
     * Whether or not to check for possible cycling
     *
     * @var boolean
     */
    protected $cycleCheck;

    /**
     * Array of visited objects; used to prevent cycling.
     *
     * @var array
     */
    protected $visited = array();

    /**
     * Constructor
     *
     * @param boolean $cycleCheck Whether or not to check for recursion when encoding
     * @return void
     */
    protected function __construct($cycleCheck = false)
    {
        $this->cycleCheck = $cycleCheck;
    }

    /**
     * Encode several classes at once
     *
     * Returns JSON encoded classes, using {@link encodeClass()}.
     *
     * @param array $classNames
     * @param string $package
     * @return string
     */
    public static function encodeClasses(array $classNames, $package = '')
    {
        $result = '';
        foreach ($classNames as $className) {
            $result .= self::encodeClass($className, $package);
        }

        return $result;
    }

    /**
     * Encodes the given $className into the class2 model of encoding PHP
     * classes into JavaScript class2 classes.
     * NOTE: Currently only public methods and variables are proxied onto
     * the client machine
     *
     * @param $className string The name of the class, the class must be
     * instantiable using a null constructor
     * @param $package string Optional package name appended to JavaScript
     * proxy class name
     * @return string The class2 (JavaScript) encoding of the class
     * @throws SZendJsonException
     */
    public static function encodeClass($className, $package = '')
    {
        $cls = new ReflectionClass($className);
        if (!$cls->isInstantiable()) {
            throw new SZendJsonException("$className must be instantiable");
        }

        return "Class.create('$package$className',{"
            .self::encodeConstants($cls).","
            .self::encodeMethods($cls).","
            .self::encodeVariables($cls).'});';
    }

    /**
     * Encode the constants associated with the ReflectionClass
     * parameter. The encoding format is based on the class2 format
     *
     * @param $cls ReflectionClass
     * @return string Encoded constant block in class2 format
     */
    private static function encodeConstants(ReflectionClass $cls)
    {
        $result = "constants : {";
        $constants = $cls->getConstants();

        $tmpArray = array();
        if (!empty($constants)) {
            foreach ($constants as $key => $value) {
                $tmpArray[] = "$key: ".self::encode($value);
            }

            $result .= implode(', ', $tmpArray);
        }

        return $result."}";
    }

    /**
     * Use the JSON encoding scheme for the value specified
     *
     * @param mixed $value The value to be encoded
     * @param boolean $cycleCheck Whether or not to check for possible object recursion when encoding
     * @return string  The encoded value
     */
    public static function encode($value, $cycleCheck = false)
    {
        $encoder = new self(($cycleCheck) ? true : false);

        return $encoder->encodeValue($value);
    }

    /**
     * Recursive driver which determines the type of value to be encoded
     * and then dispatches to the appropriate method. $values are either
     *    - objects (returns from {@link encodeObject()})
     *    - arrays (returns from {@link encodeArray()})
     *    - basic datums (e.g. numbers or strings) (returns from {@link encodeDatum()})
     *
     * @param $value mixed The value to be encoded
     * @return string Encoded value
     */
    protected function encodeValue(&$value)
    {
        if (is_object($value)) {
            return $this->encodeObject($value);
        } else {
            if (is_array($value)) {
                return $this->encodeArray($value);
            }
        }

        return $this->encodeDatum($value);
    }

    /**
     * Encode an object to JSON by encoding each of the public properties
     *
     * A special property is added to the JSON object called '__className'
     * that contains the name of the class of $value. This is used to decode
     * the object on the client into a specific class.
     *
     * @param $value object
     * @return string
     * @throws SZendJsonException If recursive checks are enabled and the object has been serialized previously
     */
    protected function encodeObject(&$value)
    {
        if ($this->cycleCheck) {
            if ($this->wasVisited($value)) {
                throw new SZendJsonException(
                    'Cycles not supported in JSON encoding, cycle introduced by '
                    .'class "'.get_class($value).'"'
                );
            }

            $this->visited[] = $value;
        }

        $props = '';
        foreach (get_object_vars($value) as $name => $propValue) {
            if (!is_null($propValue) && !empty($propValue)) {
                $props .= ','
                    .$this->encodeValue($name)
                    .':'
                    .$this->encodeValue($propValue);
            }
        }

        return '{"__className":"'.get_class($value).'"'
            .$props.'}';
    }

    /**
     * Determine if an object has been serialized already
     *
     * @param mixed $value
     * @return boolean
     */
    protected function wasVisited(&$value)
    {
        if (in_array($value, $this->visited, true)) {
            return true;
        }

        return false;
    }

    /**
     * JSON encode an array value
     *
     * Recursively encodes each value of an array and returns a JSON encoded
     * array string.
     *
     * Arrays are defined as integer-indexed arrays starting at index 0, where
     * the last index is (count($array) -1); any deviation from that is
     * considered an associative array, and will be encoded as such.
     *
     * @param $array array
     * @return string
     */
    protected function encodeArray(&$array)
    {
        $tmpArray = array();

        // Check for associative array
        if (!empty($array) && (array_keys($array) !== range(0, count($array) - 1))) {
            // Associative array
            $result = '{';
            foreach ($array as $key => $value) {
                $key = (string)$key;
                $tmpArray[] = $this->encodeString($key)
                    .':'
                    .$this->encodeValue($value);
            }
            $result .= implode(',', $tmpArray);
            $result .= '}';
        } else {
            // Indexed array
            $result = '[';
            $length = count($array);
            for ($i = 0; $i < $length; $i++) {
                $tmpArray[] = $this->encodeValue($array[$i]);
            }
            $result .= implode(',', $tmpArray);
            $result .= ']';
        }

        return $result;
    }

    /**
     * JSON encode a string value by escaping characters as necessary
     *
     * @param $value string
     * @return string
     */
    protected function encodeString(&$string)
    {
        // Escape these characters with a backslash:
        // " \ / \n \r \t \b \f
        $search = array('\\', "\n", "\t", "\r", "\b", "\f", '"');
        $replace = array('\\\\', '\\n', '\\t', '\\r', '\\b', '\\f', '\"');
        $string = str_replace($search, $replace, $string);

        // Escape certain ASCII characters:
        // 0x08 => \b
        // 0x0c => \f
        $string = str_replace(array(chr(0x08), chr(0x0C)), array('\b', '\f'), $string);

        return '"'.$string.'"';
    }

    /**
     * JSON encode a basic data type (string, number, boolean, null)
     *
     * If value type is not a string, number, boolean, or null, the string
     * 'null' is returned.
     *
     * @param $value mixed
     * @return string
     */
    protected function encodeDatum(&$value)
    {
        $result = 'null';

        if (is_int($value) || is_float($value)) {
            $result = (string)$value;
        } elseif (is_string($value)) {
            $result = $this->encodeString($value);
        } elseif (is_bool($value)) {
            $result = $value ? 'true' : 'false';
        }

        return $result;
    }

    /**
     * Encode the public methods of the ReflectionClass in the
     * class2 format
     *
     * @param $cls ReflectionClass
     * @return string Encoded method fragment
     *
     */
    private static function encodeMethods(ReflectionClass $cls)
    {
        $methods = $cls->getMethods();
        $result = 'methods:{';

        $started = false;
        foreach ($methods as $method) {
            if (!$method->isPublic() || !$method->isUserDefined()) {
                continue;
            }

            if ($started) {
                $result .= ',';
            }
            $started = true;

            $result .= ''.$method->getName().':function(';

            if ('__construct' != $method->getName()) {
                $parameters = $method->getParameters();
                $argsStarted = false;

                $argNames = "var argNames=[";
                foreach ($parameters as $param) {
                    if ($argsStarted) {
                        $result .= ',';
                    }

                    $result .= $param->getName();

                    if ($argsStarted) {
                        $argNames .= ',';
                    }

                    $argNames .= '"'.$param->getName().'"';

                    $argsStarted = true;
                }
                $argNames .= "];";

                $result .= "){"
                    .$argNames
                    .'var result = ZAjaxEngine.invokeRemoteMethod('
                    ."this, '".$method->getName()
                    ."',argNames,arguments);"
                    .'return(result);}';
            } else {
                $result .= "){}";
            }
        }

        return $result."}";
    }

    /**
     * Encode the public properties of the ReflectionClass in the class2
     * format.
     *
     * @param $cls ReflectionClass
     * @return string Encode properties list
     *
     */
    private static function encodeVariables(ReflectionClass $cls)
    {
        $properties = $cls->getProperties();
        $propValues = get_class_vars($cls->getName());
        $result = "variables:{";

        $tmpArray = array();
        foreach ($properties as $prop) {
            if (!$prop->isPublic()) {
                continue;
            }

            $tmpArray[] = $prop->getName()
                .':'
                .self::encode($propValues[$prop->getName()]);
        }
        $result .= implode(',', $tmpArray);

        return $result."}";
    }
}
