<?php
/**
 * SZend Framework
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://framework.zend.com/license/new-bsd
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@zend.com so we can send you a copy immediately.
 *
 * @author    SZend
 * @copyright  Copyright (c) 2005-2008 SZend Technologies USA Inc. (http://www.zend.com)
 * @license    http://framework.zend.com/license/new-bsd     New BSD License
 * @package    SZendJson
 * @category   SZend
 */

/**
 * SZendJson
 */
require_once dirname(__FILE__).'/../Json.php';

/**
 * SZendJsonException
 */
require_once dirname(__FILE__).'/../Json/Exception.php';

/**
 * Decode JSON encoded string to PHP variable constructs
 */
class SZendJsonDecoder
{
    /**
     * Parse tokens used to decode the JSON object. These are not
     * for public consumption, they are just used internally to the
     * class.
     */
    const EOF = 0;
    const DATUM = 1;
    const LBRACE = 2;
    const LBRACKET = 3;
    const RBRACE = 4;
    const RBRACKET = 5;
    const COMMA = 6;
    const COLON = 7;

    /**
     * Use to maintain a "pointer" to the source being decoded
     *
     * @var string
     */
    protected $source;

    /**
     * Caches the source length
     *
     * @var int
     */
    protected $sourceLength;

    /**
     * The offset within the souce being decoded
     *
     * @var int
     *
     */
    protected $offset;

    /**
     * The current token being considered in the parser cycle
     *
     * @var int
     */
    protected $token;

    /**
     * Flag indicating how objects should be decoded
     *
     * @var int
     * @access protected
     */
    protected $decodeType;

    /**
     * Constructor
     *
     * @param string $source String source to decode
     * @param int $decodeType How objects should be decoded -- see
     * {@link SZendJson::TYPE_ARRAY} and {@link SZendJson::TYPE_OBJECT} for
     * valid values
     * @return void
     */
    protected function __construct($source, $decodeType)
    {
        // Set defaults
        $this->source = $source;
        $this->sourceLength = Tools::strlen($source);
        $this->token = self::EOF;
        $this->offset = 0;

        // Normalize and set $decodeType
        if (!in_array($decodeType, array(SZendJson::TYPE_ARRAY, SZendJson::TYPE_OBJECT))) {
            $decodeType = SZendJson::TYPE_ARRAY;
        }
        $this->decodeType = $decodeType;

        // Set pointer at first token
        $this->getNextToken();
    }

    /**
     * Retrieves the next token from the source stream
     *
     * @return int Token constant value specified in class definition
     */
    protected function getNextToken()
    {
        $this->token = self::EOF;
        $this->tokenValue = null;
        $this->eatWhitespace();

        if ($this->offset >= $this->sourceLength) {
            return (self::EOF);
        }

        $str = $this->source;
        $str_length = $this->sourceLength;
        $i = $this->offset;
        $start = $i;

        switch ($str{$i}) {
            case '{':
                $this->token = self::LBRACE;
                break;
            case '}':
                $this->token = self::RBRACE;
                break;
            case '[':
                $this->token = self::LBRACKET;
                break;
            case ']':
                $this->token = self::RBRACKET;
                break;
            case ',':
                $this->token = self::COMMA;
                break;
            case ':':
                $this->token = self::COLON;
                break;
            case '"':
                $result = '';
                do {
                    $i++;
                    if ($i >= $str_length) {
                        break;
                    }

                    $chr = $str{$i};
                    if ($chr == '\\') {
                        $i++;
                        if ($i >= $str_length) {
                            break;
                        }
                        $chr = $str{$i};
                        switch ($chr) {
                            case '"':
                                $result .= '"';
                                break;
                            case '\\':
                                $result .= '\\';
                                break;
                            case '/':
                                $result .= '/';
                                break;
                            case 'b':
                                $result .= chr(8);
                                break;
                            case 'f':
                                $result .= chr(12);
                                break;
                            case 'n':
                                $result .= chr(10);
                                break;
                            case 'r':
                                $result .= chr(13);
                                break;
                            case 't':
                                $result .= chr(9);
                                break;
                            case '\'':
                                $result .= '\'';
                                break;
                            default:
                                throw new SZendJsonException("Illegal escape "
                                    ."sequence '".$chr."'");
                        }
                    } elseif ($chr == '"') {
                        break;
                    } else {
                        $result .= $chr;
                    }
                } while ($i < $str_length);

                $this->token = self::DATUM;
                //$this->tokenValue = Tools::substr($str, $start + 1, $i - $start - 1);
                $this->tokenValue = $result;
                break;
            case 't':
                if (($i + 3) < $str_length && Tools::substr($str, $start, 4) == "true") {
                    $this->token = self::DATUM;
                }
                $this->tokenValue = true;
                $i += 3;
                break;
            case 'f':
                if (($i + 4) < $str_length && Tools::substr($str, $start, 5) == "false") {
                    $this->token = self::DATUM;
                }
                $this->tokenValue = false;
                $i += 4;
                break;
            case 'n':
                if (($i + 3) < $str_length && Tools::substr($str, $start, 4) == "null") {
                    $this->token = self::DATUM;
                }
                $this->tokenValue = null;
                $i += 3;
                break;
        }

        if ($this->token != self::EOF) {
            $this->offset = $i + 1; // Consume the last token character

            return ($this->token);
        }

        $chr = $str{$i};
        if ($chr == '-' || $chr == '.' || ($chr >= '0' && $chr <= '9')) {
            $pattern = '/-?([0-9])*(\.[0-9]*)?((e|E)((-|\+)?)[0-9]+)?/s';
            if (preg_match($pattern, $str, $matches, PREG_OFFSET_CAPTURE, $start) &&
                $matches[0][1] == $start) {
                $datum = $matches[0][0];

                if (is_numeric($datum)) {
                    if (preg_match('/^0\d+$/', $datum)) {
                        throw new SZendJsonException("Octal notation not supported by JSON (value: $datum)");
                    } else {
                        $val = (int)($datum);
                        $fVal = (float)($datum);
                        $this->tokenValue = ($val == $fVal ? $val : $fVal);
                    }
                } else {
                    throw new SZendJsonException("Illegal number format: $datum");
                }

                $this->token = self::DATUM;
                $this->offset = $start + Tools::strlen($datum);
            }
        } else {
            throw new SZendJsonException('Illegal Token');
        }

        return ($this->token);
    }

    /**
     * Removes whitepsace characters from the source input
     */
    protected function eatWhitespace()
    {
        $pattern = '/([\t\b\f\n\r ])*/s';
        if (preg_match($pattern, $this->source, $matches, PREG_OFFSET_CAPTURE, $this->offset) &&
            $matches[0][1] == $this->offset) {
            $this->offset += Tools::strlen($matches[0][0]);
        }
    }

    /**
     * Decode a JSON source string
     *
     * Decodes a JSON encoded string. The value returned will be one of the
     * following:
     *        - integer
     *        - float
     *        - boolean
     *        - null
     *      - StdClass
     *      - array
     *         - array of one or more of the above types
     *
     * By default, decoded objects will be returned as associative arrays; to
     * return a StdClass object instead, pass {@link SZendJson::TYPE_OBJECT} to
     * the $objectDecodeType parameter.
     *
     * Throws a SZendJsonException if the source string is null.
     *
     * @static
     * @access public
     * @param string $source String to be decoded
     * @param int $objectDecodeType How objects should be decoded; should be
     * either or {@link SZendJson::TYPE_ARRAY} or
     * {@link SZendJson::TYPE_OBJECT}; defaults to TYPE_ARRAY
     * @return mixed
     * @throws SZendJsonException
     */
    public static function decode($source = null, $objectDecodeType = SZendJson::TYPE_ARRAY)
    {
        if (null === $source) {
            throw new SZendJsonException('Must specify JSON encoded source for decoding');
        } elseif (!is_string($source)) {
            throw new SZendJsonException('Can only decode JSON encoded strings');
        }

        $decoder = new self($source, $objectDecodeType);

        return $decoder->decodeValue();
    }

    /**
     * Recursive driving rountine for supported toplevel tops
     *
     * @return mixed
     */
    protected function decodeValue()
    {
        switch ($this->token) {
            case self::DATUM:
                $result = $this->tokenValue;
                $this->getNextToken();

                return ($result);
            case self::LBRACE:
                return ($this->decodeObject());
            case self::LBRACKET:
                return ($this->decodeArray());
            default:
                return null;
        }
    }

    /**
     * Decodes an object of the form:
     *  { "attribute: value, "attribute2" : value,...}
     *
     * If ZJsonEnoder or ZJAjax was used to encode the original object
     * then a special attribute called __className which specifies a class
     * name that should wrap the data contained within the encoded source.
     *
     * Decodes to either an array or StdClass object, based on the value of
     * {@link $decodeType}. If invalid $decodeType present, returns as an
     * array.
     *
     * @return array|StdClass
     */
    protected function decodeObject()
    {
        $members = array();
        $tok = $this->getNextToken();

        while ($tok && $tok != self::RBRACE) {
            if ($tok != self::DATUM || !is_string($this->tokenValue)) {
                throw new SZendJsonException('Missing key in object encoding: '.$this->source);
            }

            $key = $this->tokenValue;
            $tok = $this->getNextToken();

            if ($tok != self::COLON) {
                throw new SZendJsonException('Missing ":" in object encoding: '.$this->source);
            }

            $tok = $this->getNextToken();
            $members[$key] = $this->decodeValue();
            $tok = $this->token;

            if ($tok == self::RBRACE) {
                break;
            }

            if ($tok != self::COMMA) {
                throw new SZendJsonException('Missing "," in object encoding: '.$this->source);
            }

            $tok = $this->getNextToken();
        }

        switch ($this->decodeType) {
            case SZendJson::TYPE_OBJECT:
                // Create new StdClass and populate with $members
                $result = new StdClass();
                foreach ($members as $key => $value) {
                    $result->$key = $value;
                }
                break;
            case SZendJson::TYPE_ARRAY:
            default:
                $result = $members;
                break;
        }

        $this->getNextToken();

        return $result;
    }

    /**
     * Decodes a JSON array format:
     *    [element, element2,...,elementN]
     *
     * @return array
     */
    protected function decodeArray()
    {
        $result = array();
        $tok = $this->getNextToken(); // Move past the '['
        $index = 0;

        while ($tok && $tok != self::RBRACKET) {
            $result[$index++] = $this->decodeValue();

            $tok = $this->token;

            if ($tok == self::RBRACKET || !$tok) {
                break;
            }

            if ($tok != self::COMMA) {
                throw new SZendJsonException('Missing "," in array encoding: '.$this->source);
            }

            $tok = $this->getNextToken();
        }

        $this->getNextToken();

        return ($result);
    }
}
