<?php
/**
 * Client
 *
 * @author      Shopimind <contact@shopimind.com>
 * @copyright   Copyright (c) 2013 - IDVIVE SARL (http://www.idvive.com)
 * @license     New BSD license (http://license.idvive.com)
 * @package     ShopimindClient
 * @version     $Id Client.php 2013-04-24$
 */

require_once dirname(__FILE__).'/library/SZend/Http/Client.php';
require_once dirname(__FILE__).'/library/SZend/Json.php';
require_once dirname(__FILE__).'/../bin/Configuration.php';

final class ShopimindClientClient
{
    protected $reminders = array(
        'getDroppedOutCart' => 1,
        'getBirthdayClients' => 2,
        'getGoodClientsByAmount' => 4,
        'getGoodClientsByNumberOrders' => 8,
        'getMissingClients' => 16,
        'getOrdersByStatus' => 32,
        'getVoucherUnused' => 64,
        'syncOrders' => 128,
        'getBirthdayClientsSignUp' => 256,
        'getInactiveClients' => 512,
    );

    /**
     * Permet d'envoyer des requêtes au serveur
     *
     * @var SZendHttpClient
     */
    private $client;

    private $restService;

    public function __construct()
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        $url = $config['api']['url'];

        $this->setClient(new SZendHttpClient($url));
        $this->initHeaders();
    }

    /**
     * Permet d'initialiser le header avec des valeurs par défaut
     *
     * @return void
     */
    public function initHeaders()
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        $this->getClient()->setHeaders($config['header']['client_id'], $config['api']['identifiant'])
            ->setHeaders($config['header']['build'], $this->getBuild())
            ->setHeaders($config['header']['version'], $config['version'])
            ->setHeaders('version', $config['version']);
    }

    /**
     * Connecteur HTTP
     *
     * @return SZendHttpClient
     */
    public function getClient()
    {
        return $this->client;
    }

    /**
     * Permet de modifier le connecteur HTTP
     *
     * @param SZendHttpClient $client
     * @return ShopimindClientClient
     */
    public function setClient($client)
    {
        $this->client = $client;

        return $this;
    }

    /**
     * Permet de récupérer la build du client
     *
     * @return int
     */
    public function getBuild()
    {
        $build = 0;

        if (file_exists(dirname(__FILE__).'/../callback.php')) {
            require_once dirname(__FILE__).'/../callback.php';

            foreach (get_class_methods('ShopimindClientCallback') as $method) {
                if (array_key_exists($method, $this->reminders)) {
                    $build += $this->reminders[$method];
                }
            }
        }

        return $build;
    }

    /**
     * Permet d'ajouter des paramètres get à la requête
     *
     * @return ShopimindClientClient
     */
    public function setParameterGet($name, $value = null)
    {
        $this->getClient()->setParameterGet($name, $value);

        return $this;
    }

    /**
     * Permet de tester la bonne communication avec le serveur
     *
     * @return boolean
     */
    public function sayHello()
    {
        $this->setParameterPost('testCommunication', time());
        $response = $this->sendRequest('POST');

        if (is_array($response) && array_key_exists('success', $response)) {
            return $response['success'];
        }

        return false;
    }

    /**
     * Permet d'ajouter des paramètres post à la requête
     *
     * @return ShopimindClientClient
     */
    public function setParameterPost($name, $value = null)
    {
        $this->getClient()->setParameterPost($name, $value);

        return $this;
    }

    /**
     * Permet d'envoyer la requête
     *
     * @return SZendHttpResponse
     */
    public function sendRequest($method = 'GET')
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        $this->getClient()->setMethod($method);
        $this->getClient()->setConfig(array('timeout' => 5));
        $token = $this->getToken();
        $this->getClient()->setHeaders($config['header']['hmac'], $token);
        $this->getClient()->setHeaders($config['header']['promo_code'], $config['promo_code']);

        if ($this->getRestService() !== null) {
            $uri = $this->getClient()->getUri(true);
            $uri .= '/'.$this->getRestService();
            $this->getClient()->setUri($uri);
        }

        $response = $this->getClient()->request();

        return SZendJson::decode($response->getBody());
    }


    private function customHmac($algo, $data, $key, $raw_output = false)
    {
        $algo = Tools::strtolower($algo);
        $pack = 'H'.Tools::strlen(hash($algo, 'test'));
        $size = 64;
        $opad = str_repeat(chr(0x5C), $size);
        $ipad = str_repeat(chr(0x36), $size);

        if (Tools::strlen($key) > $size) {
            $key = str_pad(pack($pack, hash($algo, $key)), $size, chr(0x00));
        } else {
            $key = str_pad($key, $size, chr(0x00));
        }

        for ($i = 0; $i < Tools::strlen($key) - 1; $i++) {
            $opad[$i] = $opad[$i] ^ $key[$i];
            $ipad[$i] = $ipad[$i] ^ $key[$i];
        }

        $output = hash($algo, $opad.pack($pack, hash($algo, $ipad.$data)));

        return ($raw_output) ? pack($pack, $output) : $output;
    }


    /**
     * Permet de récupérer le token
     *
     * @return string
     */
    public function getToken()
    {
        $config = ShopimindClientConfiguration::getGlobalConfig();

        $params = $this->getClient()->getParams();

        if ($params !== null) {
            uksort($params, 'strnatcasecmp');
            $params = $this->implodeRecursive($params);
        }

        $password = sha1($config['api']['password']);

        return $this->customHmac('sha256', $params, $password);
    }

    private function implodeRecursive(array $array, $glue = ';')
    {
        $glued_string = '';
        // Recursively iterates array and adds key/value to glued string
        array_walk_recursive(
            $array,
            array($this, 'formatString'),
            array('glue' => $glue, 'glued_string' => &$glued_string)
        );
        // Removes last $glue from string
        Tools::strlen($glue) > 0 and $glued_string = Tools::substr($glued_string, 0, -Tools::strlen($glue));

        // Trim ALL whitespace
        $glued_string = preg_replace("/(\s)/ixsm", '', $glued_string);

        return (string)$glued_string;
    }

    /**
     * Permet de récupérer le service à appeler
     *
     * @return string
     */
    public function getRestService()
    {
        return $this->restService;
    }

    /**
     * Permet de définir le service à appeler
     *
     * @param string $restService
     * @return ShopimindClientClient
     */
    public function setRestService($restService)
    {
        $this->restService = $restService;

        return $this;
    }

    protected function formatString($value, $key, $options)
    {
        if (!empty($value)) {
            $options['glued_string'] .= $key.$options['glue'];
            $options['glued_string'] .= $value.$options['glue'];
        }
    }
}
