<?php
/**
 * Callback
 *
 * @author Shopimind <contact@shopimind.com>
 * @copyright Copyright (c) 2013 - IDVIVE SARL (http://www.idvive.com)
 * @license New BSD license (http://license.idvive.com)
 * @package ShopimindClient
 * @version $Id callback.php 2013-04-23$
 */

if (!isset($this) || (property_exists($this, 'name') && $this->name !== 'shopimind')) {
    require_once(dirname(__FILE__) . '/../../../config/config.inc.php');

    if (file_exists(dirname(__FILE__) . '/../../../init.php')) {
        require_once(dirname(__FILE__) . '/../../../init.php');
    }
    if (! class_exists('ShopiMind', false)) {
        require_once(dirname(__FILE__) . '/../shopimind.php');
    }
}
if (file_exists(dirname(__FILE__) . '/callback_override.php')) {
    require_once(dirname(__FILE__) . '/callback_override.php');
}

class ShopimindClientCallback
{
    /**
     * Permet de récupérer la liste des statuts de commandes
     * Retour attendu : Un tableau de liste de statuts
     * @return array
     */
    public static function getOrdersStatus()
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();
        if (_PS_VERSION_ < "1.4.7" || (_PS_VERSION_ >= "1.5.0.0" && _PS_VERSION_ < "1.5.0.5")) {
            $query = 'SELECT os.`id_order_state`, osl.`name`
            FROM `'._DB_PREFIX_.'order_state` os
            LEFT JOIN `'._DB_PREFIX_.'order_state_lang` osl ON (os.`id_order_state` = osl.`id_order_state` 
            AND osl.`id_lang` = '.(int)Configuration::get('PS_LANG_DEFAULT').') ORDER BY `name` ASC';
        } else {
            $query = 'SELECT os.`id_order_state`, osl.`name`
            FROM `'._DB_PREFIX_.'order_state` os
            LEFT JOIN `'._DB_PREFIX_.'order_state_lang` osl ON (os.`id_order_state` = osl.`id_order_state`
            AND osl.`id_lang` = '.(int)Configuration::get('PS_LANG_DEFAULT').')
            WHERE deleted = 0 ORDER BY `name` ASC';
        }
        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[$row['id_order_state']] = $row['name'];
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer la liste des fabricants présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncManufacturers(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_manufacturer = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $id_manufacturer = (!is_array($id_manufacturer) &&
            $id_manufacturer > 0) ? array($id_manufacturer) : $id_manufacturer;

        $query = 'SELECT '.($justCount ? 'COUNT(a.`id_manufacturer`) AS `nbManufacturers`' :
                'a.`id_manufacturer`, a.`name`, a.`date_add`, a.`date_upd`, a.`active`'.(self::isMultiShop($id_shop) ?
                    ', manufacturer_shop.id_shop' : ', NULL AS id_shop')).'
            FROM `'._DB_PREFIX_.'manufacturer` a
			'.(self::isMultiShop($id_shop) ? Shop::addSqlAssociation('manufacturer', 'a') : '').'
            WHERE '.($id_manufacturer && is_array($id_manufacturer) ?
                'a.`id_manufacturer` IN ('.implode(',', array_map('intval', $id_manufacturer)).')' : '1').'
            '.($lastUpdate ? ' AND a.`date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
			'.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '').'
            '.($justCount ? '' : 'ORDER BY `date_upd`').'
            '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if ($justCount) {
            return array(
                'count' => $results[0]['nbManufacturers'],
            );
        }
        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[] = array(
                    'id_manufacturer' => $row['id_manufacturer'],
                    'id_shop' => $row['id_shop'],
                    'name' => $row['name'],
                    'active' => (int)$row['active'],
                    'date_creation' => $row['date_add'],
                    'date_update' => $row['date_upd'],
                );
            }
        }

        return $return;
    }



    /**
     * *************** **
     * SYNC REQUESTS *
     * * *************** *
     */

    public static function getContext($id_shop = false)
    {
        $module = new ShopiMind();
        $context = $module->getContext();
        if ($id_shop && !is_null($context->shop) && $context->shop->id && self::isMultiShop($id_shop)) {
            Shop::setContext(1, $id_shop);
            $shop = new Shop($id_shop);
            Context::getContext()->shop = $shop;
            $context->shop = $shop;
        }

        return $context;
    }

    public static function isMultiShop($id_shop)
    {
        return (bool)($id_shop !== null &&
            (int)$id_shop &&
            _PS_VERSION_ >= 1.5 &&
            Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE'));
    }

    /**
     * Permet de récupérer la liste des transporteurs présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncCarriers(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_carrier = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();
        $lastUpdate = isset($lastUpdate) ? $lastUpdate : false;

        $id_carrier = (!is_array($id_carrier) && $id_carrier > 0) ? array($id_carrier) : $id_carrier;

        $query = 'SELECT '.($justCount ? 'COUNT(a.`id_carrier`) AS `nbCarriers`' :
                'a.`id_carrier`, a.`name`, a.`active`'.(self::isMultiShop($id_shop) ?
                    ', carrier_shop.id_shop' : ', NULL AS id_shop')).'
            FROM `'._DB_PREFIX_.'carrier` a
			'.(self::isMultiShop($id_shop) ? Shop::addSqlAssociation('carrier', 'a') : '').'
            WHERE '.($id_carrier && is_array($id_carrier) ?
                'a.`id_carrier` IN ('.implode(',', array_map('intval', $id_carrier)).')' :
                '1').'
			'.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '').'
            '.($justCount ? '' : 'ORDER BY `id_carrier`').'
            '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if ($justCount) {
            return array(
                'count' => $results[0]['nbCarriers'],
            );
        }
        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[] = array(
                    'id_carrier' => $row['id_carrier'],
                    'id_shop' => $row['id_shop'],
                    'name' => $row['name'],
                    'active' => (int)$row['active'],
                );
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer la liste des statuts de commandes présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncOrdersStatus(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_status = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $id_shop = isset($id_shop) ? $id_shop : 0;
        $lastUpdate = isset($lastUpdate) ? $lastUpdate : false;

        $id_status = (!is_array($id_status) && $id_status > 0) ? array($id_status) : $id_status;

        $query = 'SELECT '.($justCount ? 'COUNT(a.`id_order_state`) AS `nbOrdersStatus`' :
        'a.`id_order_state`, a.`deleted`, b.`name`, b.`id_lang`').'
        FROM `'._DB_PREFIX_.'order_state` a
        JOIN `'._DB_PREFIX_.'order_state_lang` b ON (b.`id_order_state` = a.`id_order_state`)
        WHERE '.($id_status && is_array($id_status) ? 'a.`id_order_state` IN ('.implode(',', array_map('intval', $id_status)).')' : '1').'
        '.($justCount ? '' : 'ORDER BY `id_order_state`').'
        '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if ($justCount) {
            return array(
                'count' => $results[0]['nbOrdersStatus'],
            );
        }
        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $lang_status = Language::getLanguage($row['id_lang']);
                $lang = Language::getIsoById($row['id_lang']);

                if (!$lang) {
                    continue;
                }

                if (is_array($lang_status) && array_key_exists('active', $lang_status) && $lang_status['active'] == 0) {
                    continue;
                }

                $return[] = array(
                    'id_status' => $row['id_order_state'],
                    'name' => $row['name'],
                    'lang' => $lang,
                    'deleted' => (int)$row['deleted'],
                );
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer la liste des groupes de contacts présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncCustomersGroups(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_group = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $id_group = (!is_array($id_group) && $id_group > 0) ? array($id_group) : $id_group;

        $query = 'SELECT '.($justCount ? 'COUNT(a.`id_group`) AS `nbGroups`' :
                'a.`id_group`, a.`date_add`, a.`date_upd`, b.`id_lang`, b.`name`'.(self::isMultiShop($id_shop) ?
                    ', group_shop.id_shop' : ', NULL AS id_shop')).'
            FROM `'._DB_PREFIX_.'group` a
			JOIN `'._DB_PREFIX_.'group_lang` b ON (b.`id_group` = a.`id_group` AND 
			    b.`id_lang` = '.(int)Configuration::get('PS_LANG_DEFAULT').')
			'.(self::isMultiShop($id_shop) ? Shop::addSqlAssociation('group', 'a') : '').'
            WHERE '.($id_group && is_array($id_group) ? 'a.`id_group` IN ('.implode(',', array_map('intval', $id_group)).')' : '1').'
            '.($lastUpdate ? ' AND a.`date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
			'.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '').'
            '.($justCount ? '' : 'ORDER BY `date_upd`').'
            '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if ($justCount) {
            return array(
                'count' => $results[0]['nbGroups'],
            );
        }
        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $lang = Language::getIsoById($row['id_lang']);

                $return[] = array(
                    'id_group' => $row['id_group'],
                    'id_shop' => $row['id_shop'],
                    'name' => $row['name'],
                    'lang' => $lang,
                    'date_creation' => $row['date_add'],
                    'date_update' => $row['date_upd'],
                );
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer la liste des contacts présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncCustomers(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_customer = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $id_customer = (!is_array($id_customer) && $id_customer > 0) ? array($id_customer) : $id_customer;

        $query = 'SELECT '.($justCount ? 'COUNT(a.`id_customer`) AS `nbCutomers`' : 'a.`id_customer`').'
        FROM `'._DB_PREFIX_.'customer` a
        WHERE '.($id_customer && is_array($id_customer) ?
                '`id_customer` IN ('.implode(',', array_map('intval', $id_customer)).')' :
                '1').'
        '.($lastUpdate ? ' AND a.`date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
        '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false, 'a') : '').'
        '.($justCount ? '' : 'ORDER BY `date_upd`').'
        '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if ($justCount) {
            return array(
                'count' => $results[0]['nbCutomers'],
            );
        }
        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[] = self::getUser($row['id_customer']);
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer les informations utilisateur(s)
     * en fonction de son identifiant
     *
     * Paramètre : $id -> L'ID de l'utilisateur ou liste d'utilisateurs
     *
     * Retour attendu : Tableau attendu (ou tableau de tableaux) :
     * [
     * 'last_name',
     * 'first_name',
     * 'email_address',
     * 'gender',(1 = homme, 2 = femme, 0 = inconnu)(optionnel)
     * 'locale'
     * ]
     *
     * @param mixed|array $id
     * @return array
     */
    public static function getUser($id)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();
        if (_PS_VERSION_ < '1.5.4') {
            $query = '
            SELECT a.`active`, a.`deleted`, a.`optin`, a.`newsletter`, a.`birthday`, a.`date_add`,
             a.`date_upd`, a.`id_customer`, a.`id_gender`, a.`lastname`, a.`firstname`,
              a.`email`,c.`iso_code`, SUBSTRING(d.`accept_language`,1,2) AS `iso_code2`,
               `country`.`iso_code` as `country_code`'.
            (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                ', a.`id_shop`' :
                ', "0" AS `id_shop`').'
            FROM `'._DB_PREFIX_.'customer` a
            LEFT JOIN `'._DB_PREFIX_.'address` AS `address` ON (`address`.`id_customer` = `a`.`id_customer`)
            LEFT JOIN `'._DB_PREFIX_.'country` AS `country` ON (`address`.`id_country` = `country`.`id_country`)
            LEFT JOIN `'._DB_PREFIX_.'cart` b ON (a.`id_customer` = b.`id_customer`)
            LEFT JOIN `'._DB_PREFIX_.'lang` c ON (b.`id_lang` = c.`id_lang`)
            LEFT JOIN `'._DB_PREFIX_.'guest` d ON (a.`id_customer` = d.`id_customer`)
            WHERE a.`id_customer` IN('.(implode(', ', array_map('pSQL', array_map('intval', (array)$id)))).')
            GROUP BY a.`email`';
        } else {
            $query = '
            SELECT a.`active`, a.`deleted`, a.`optin`, a.`newsletter`, a.`birthday`, a.`date_add`, a.`date_upd`,
             a.`id_customer`, a.`id_gender`, a.`lastname`, a.`firstname`, a.`email`, b.`iso_code`,
              `country`.`iso_code` as `country_code` '.
                (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                ', a.`id_shop`' :
                ', "0" AS `id_shop`').'
            FROM `'._DB_PREFIX_.'customer` a
            LEFT JOIN `'._DB_PREFIX_.'address` AS `address` ON (`address`.`id_customer` = `a`.`id_customer`)
            LEFT JOIN `'._DB_PREFIX_.'country` AS `country` ON (`address`.`id_country` = `country`.`id_country`)
            LEFT JOIN `'._DB_PREFIX_.'lang` b ON (a.`id_lang` = b.`id_lang`)
            WHERE a.`id_customer` IN('.(implode(', ', array_map('pSQL', array_map('intval', (array)$id)))).')
            GROUP BY a.`email`';
        }

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if ($results && is_array($results) && count($results)) {
            foreach ($results as $row) {
                // Mlle = Mme
                if ((int)$row['id_gender'] == 3) {
                    $row['id_gender'] = 2;
                }
                $return[] = array(
                    'id_customer' => $row['id_customer'],
                    'shop_id_shop' => $row['id_shop'],
                    'locale' => self::getLocaleFromSqlRow($row),
                    'first_name' => $row['firstname'],
                    'last_name' => $row['lastname'],
                    'birthday' => (array_key_exists('birthday', $row) ? $row['birthday'] : 0),
                    'email' => $row['email'],
                    'optin' => $row['optin'],
                    'newsletter' => $row['newsletter'],
                    'customer_since' => $row['date_add'],
                    'customer_update' => $row['date_upd'],
                    'gender' => (array_key_exists('id_gender', $row) &&
                    ($row['id_gender'] == 1 || $row['id_gender'] == 2) ?
                        $row['id_gender'] : 0),
                    'groups' => self::getSpecificCustomerGroups($row['id_customer']),
                    'active' => ((int)$row['deleted'] === 1 || (int)$row['active'] === 0) ? 0 : 1,
                    'addresses' => self::getSpecificCustomerAddresses($row['id_customer']),
                    'custom' => array(),
                );
            }
        }

        return (sizeof($return) === 1 ? $return[0] : $return);
    }

    public static function getLocaleFromSqlRow($row)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        if (array_key_exists('iso_code', $row) && $row['iso_code']) {
            $lang = $row['iso_code'];
        } elseif (array_key_exists('iso_code2', $row) && $row['iso_code2']) {
            $lang = $row['iso_code2'];
        } else {
            $lang = Language::getIsoById(Configuration::get('PS_LANG_DEFAULT'));
        }

        if (array_key_exists('country_code', $row) && $row['country_code']) {
            $country = $row['country_code'];
        } else {
            $country = '00';
        }

        return Tools::strtolower($lang).'_'.Tools::strtoupper($country);
    }

    public static function getSpecificCustomerGroups($id_customer)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $return = array();
        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
            SELECT `id_group`
            FROM `'._DB_PREFIX_.'customer_group`
            WHERE `id_customer` = '.(int)$id_customer);

        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[] = $row['id_group'];
            }
        }

        return $return;
    }

    public static function getSpecificCustomerAddresses($id_customer)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
            SELECT DISTINCT
                `address`.`id_address`,
                `address`.`firstname`,
                `address`.`lastname`,
                `address`.`phone` AS `phone1`,
                `address`.`phone_mobile` AS `phone2`,
                `address`.`city`,
                `address`.`company`,
                `address`.`address1`,
                `address`.`address2`,
                `address`.`postcode`,
                `address`.`other`,
                `address`.`active`,
                `address`.`deleted`,
                `country`.`iso_code` AS `country_code`
            FROM `'._DB_PREFIX_.'address` AS `address`
            LEFT JOIN `'._DB_PREFIX_.'country` AS `country` ON (`address`.`id_country` = `country`.`id_country`)
            WHERE `address`.`id_customer` = '.(int)$id_customer);

        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[] = array(
                    'id_address' => (array_key_exists('id_address', $row) && $row['id_address'] ?
                        $row['id_address'] :
                        ''),
                    'first_name' => (array_key_exists('firstname', $row) && $row['firstname'] ?
                        $row['firstname'] :
                        ''),
                    'last_name' => (array_key_exists('lastname', $row) && $row['lastname'] ?
                        $row['lastname'] :
                        ''),
                    'phone1' => (array_key_exists('phone1', $row) && $row['phone1'] ?
                        $row['phone1'] :
                        ''),
                    'phone2' => (array_key_exists('phone2', $row) && $row['phone2'] ?
                        $row['phone2'] :
                        ''),
                    'company' => (array_key_exists('company', $row) && $row['company'] ?
                        $row['company'] :
                        ''),
                    'address1' => (array_key_exists('address1', $row) && $row['address1'] ?
                        $row['address1'] :
                        ''),
                    'address2' => (array_key_exists('address2', $row) && $row['address2'] ?
                        $row['address2'] :
                        ''),
                    'postcode' => (array_key_exists('postcode', $row) && $row['postcode'] ?
                        $row['postcode'] :
                        ''),
                    'city' => (array_key_exists('city', $row) && $row['city'] ?
                        $row['city'] :
                        ''),
                    'other' => (array_key_exists('other', $row) && $row['other'] ?
                        $row['other'] :
                        ''),
                    'active' => ((array_key_exists('active', $row) &&
                        array_key_exists('deleted', $row)) ?
                        ((int)$row['active'] & !(int)$row['deleted']) :
                        false)
                );
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer la liste des contacts abonnés NL présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncNLSubscribers(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $return = array();

        // Si la fonctionnalité est désactivée, on renvoie un tableau vide
        $isEnabled = Configuration::get('SPM_syncNL');
        if (!$isEnabled) {
            return $return;
        }

        // Récupération de la configuration
        $table = Configuration::get('SPM_syncNL_table');
        $fields = explode('|', Configuration::get('SPM_syncNL_table_fields'));

        $tableFields = $finalFields = array();
        if (is_array($fields) && count($fields) > 0) {
            foreach ($fields as $field) {
                $field = explode('-', $field);
                $tableFields[$field[1]] = $field[0];
                array_push($finalFields, $field[1]);
            }
        } else {
            // Si l'utilisateur n'a choisi aucun champ, on renvoie un tableau vide
            return $return;
        }

        $query = 'SELECT '.($justCount ?
                'COUNT(*) AS `nbSubscribers`' :
                '`'.implode('`, `', array_map('bqSQL', $tableFields)).'`').'
            FROM `'.$table.'` a
            WHERE 1
            '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false, 'a') : '').'
            '.($lastUpdate && array_key_exists('last_update', $tableFields) ?
                ' AND a.`'.$tableFields['last_update'].'` >= "'.pSQL($lastUpdate).'"' :
                '').'
            '.($justCount ? '' : (array_key_exists('last_update', $tableFields) ?
                'ORDER BY a.`'.$tableFields['last_update'].'`' :
                '')).'
            '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if ($justCount) {
            return array(
                'count' => $results[0]['nbSubscribers'],
            );
        }

        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                // On teste s'il s'agit bien d'une adresse e-mail dans le champ rattaché à l'email
                if (array_key_exists($tableFields['email'], $row) &&
                    !filter_var($row[$tableFields['email']], FILTER_VALIDATE_EMAIL)) {
                    continue;
                }

                array_push($return, array(
                    'id_shop' => array_key_exists($tableFields['id_shop'], $row) ?
                        $row[$tableFields['id_shop']] :
                        null,
                    'last_name' => array_key_exists($tableFields['last_name'], $row) ?
                        $row[$tableFields['last_name']] :
                        null,
                    'first_name' => array_key_exists($tableFields['first_name'], $row) ?
                        $row[$tableFields['first_name']] :
                        null,
                    'email' => array_key_exists($tableFields['email'], $row) ?
                        $row[$tableFields['email']] :
                        null,
                    'lang' => array_key_exists($tableFields['lang'], $row) ?
                        Language::getIsoById($row[$tableFields['lang']]) :
                        null,
                    'zipcode' => array_key_exists($tableFields['zipcode'], $row) ?
                        $row[$tableFields['zipcode']] :
                        null,
                    'last_update' => array_key_exists($tableFields['last_update'], $row) ?
                        $row[$tableFields['last_update']] :
                        null,
                    'newsletter' => (array_key_exists($tableFields['status'], $row) &&
                        $row[$tableFields['status']] == 1) ?
                        $row[$tableFields['status']] :
                        0,
                ));
            }
        }

        return $return;
    }

    /**
     * Permet de récupérer la liste des bons de réduction présents dans la base
     * Cette méthode est utilisée pour la fonctionnalité de campagne SMS et Email
     */
    public static function syncVouchers(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_cart_rule = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $id_cart_rule = (!is_array($id_cart_rule) && $id_cart_rule != '') ? array($id_cart_rule) : $id_cart_rule;

        if (_PS_VERSION_ >= 1.5) {
            $query = 'SELECT '.($justCount ? 'COUNT(a.`id_cart_rule`) AS `nbVouchers`' :
            'a.*, crl.*'.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ? ', crs.id_shop' :
                ', NULL AS id_shop')).'
            FROM `'._DB_PREFIX_.'cart_rule` a
            LEFT JOIN '._DB_PREFIX_.'cart_rule_shop crs ON a.id_cart_rule = crs.id_cart_rule
            LEFT JOIN '._DB_PREFIX_.'cart_rule_lang crl ON a.id_cart_rule = crl.id_cart_rule
            WHERE '.($id_cart_rule && is_array($id_cart_rule) ?
                    '(a.`id_cart_rule` IN ("'.implode('","', array_map('pSQL', $id_cart_rule)).'") 
                    OR a.`code` IN ("'.implode('","', array_map('pSQL', $id_cart_rule)).'"))' : '1').'
            '.($lastUpdate ? ' AND a.`date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
            AND (a.shop_restriction = 0'.((self::isMultiShop($id_shop)) ? ' OR crs.id_shop = '.(int)$id_shop : '').')
            '.($justCount ? '' : 'ORDER BY `date_upd`, crl.`id_lang`').'
            '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

            $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

            if ($justCount) {
                return array(
                    'count' => $results[0]['nbVouchers'],
                );
            }

            if ($results && is_array($results) && sizeof($results)) {
                foreach ($results as $row) {
                    // On détermine le type du bon de réduction
                    $typeVoucher = 0;
                    $amount = 0;

                    $freeShipping = $row['free_shipping'] == 1;
                    $reduction = ($row['reduction_percent'] > 0 || $row['reduction_amount'] > 0);
                    $gift = $row['gift_product'] > 0;

                    if (($freeShipping && $reduction) || ($freeShipping && $gift) || ($reduction && $gift) || $gift) {
                        $typeVoucher = 4; // Custom
                        if ($row['reduction_amount'] > 0) {
                            $amount = $row['reduction_amount'];
                        } elseif ($row['reduction_percent'] > 0) {
                            $amount = $row['reduction_percent'];
                        }
                    } elseif ($freeShipping) {
                        // Frais de port offerts
                        $typeVoucher = 3;
                    } elseif ($row['reduction_amount'] > 0) {
                        // Montant
                        $typeVoucher = 2;
                        $amount = $row['reduction_amount'];
                    } elseif ($row['reduction_percent'] > 0) {
                        // Pourcentage
                        $typeVoucher = 1;
                        $amount = $row['reduction_percent'];
                    }

                    $currency = new Currency($row['minimum_amount_currency']);
                    $currency2 = new Currency($row['reduction_currency']);
                    $keyLang = $row['id_cart_rule'].'-'.Language::getIsoById($row['id_lang']);

                    if (!array_key_exists($keyLang, $return)) {
                        $return[$keyLang] = array(
                            'id_shop' => $row['id_shop'],
                            'id_voucher' => $row['id_cart_rule'],
                            'code' => $row['code'],
                            'name' => trim($row['description']) != '' ? $row['description'] : $row['name'],
                            'date_from' => $row['date_from'],
                            'date_to' => $row['date_to'],
                            'active' => $row['active'],
                            'id_customer_shop' => $row['id_customer'],
                            'date_creation' => $row['date_add'],
                            'date_update' => $row['date_upd'],
                            'type_voucher' => $typeVoucher,
                            'amount' => $amount,
                            'reduction_tax' => $row['reduction_tax'],
                            'reduction_currency' => $currency2->iso_code,
                            'minimum_amount' => $row['minimum_amount'],
                            'minimum_amount_currency' => $currency->iso_code,
                            'used' => $row['quantity'] == 0 ? 1 : 0,
                            'lang' => Language::getIsoById($row['id_lang']),
                        );
                    }
                }
            }
        } else {
            // Version inférieure à 1.5
            $query = 'SELECT '.($justCount ? 'COUNT(a.`id_discount`) AS `nbVouchers`' : 'a.*, NULL AS id_shop').'
				FROM `'._DB_PREFIX_.'discount` a
				WHERE '.($id_cart_rule && is_array($id_cart_rule) ?
                    '(a.`discount` IN ("'.implode('","', array_map('pSQL', $id_cart_rule)).'") 
                    OR a.`name` IN ("'.implode('","', array_map('pSQL', $id_cart_rule)).'"))'
                    : '1').'
				/*AND a.date_to >= "'.date('Y-m-d H:i:s').'"*/
				'.($lastUpdate ? ' AND a.`date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
				'.($justCount ? '' : 'ORDER BY `date_upd`').'
				'.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

            $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

            if ($justCount) {
                return array(
                    'count' => $results[0]['nbVouchers'],
                );
            }

            if ($results && is_array($results) && sizeof($results)) {
                foreach ($results as $row) {
                    $query = 'SELECT crl.*
							FROM `'._DB_PREFIX_.'discount` a
							LEFT JOIN '._DB_PREFIX_.'discount_lang crl ON a.id_discount = crl.id_discount
							WHERE a.id_discount = '.(int)($row['id_discount']);

                    $lang = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

                    $arrayLang = array();
                    $description = null;
                    if ($lang && is_array($lang) && sizeof($lang)) {
                        foreach ($lang as $l) {
                            $arrayLang[] = array(
                                'lang' => Language::getIsoById($l['id_lang']),
                                'name' => $l['description'],
                            );

                            if (is_null($description)) {
                                $description = $l['description'];
                            }
                        }
                    }

                    // On détermine le type du bon de réduction
                    $typeVoucher = 0;
                    $amount = 0;

                    if ($row['id_discount_type'] == 1) {
                        $typeVoucher = 1; // Pourcentage
                        $amount = $row['value'];
                    } elseif ($row['id_discount_type'] == 2) {
                        $typeVoucher = 2; // Montant
                        $amount = $row['value'];
                    } elseif ($row['id_discount_type'] == 3) {
                        $typeVoucher = 3; // Frais de port offerts
                    } else {
                        $typeVoucher = 4; // Custom
                    }


                    $currency = new Currency($row['id_currency']);
                    $return[] = array(
                        'id_shop' => null,
                        'id_voucher' => $row['id_discount'],
                        'code' => $row['name'],
                        'name' => $row['description'],
                        'date_from' => $row['date_from'],
                        'date_to' => $row['date_to'],
                        'active' => $row['active'],
                        'id_customer_shop' => $row['id_customer'],
                        'date_creation' => $row['date_add'],
                        'date_update' => $row['date_upd'],
                        'type_voucher' => $typeVoucher,
                        'amount' => $amount,
                        'reduction_tax' => $row['include_tax'],
                        'reduction_currency' => $currency->iso_code,
                        'minimum_amount' => $row['minimal'],
                        'minimum_amount_currency' => $currency->iso_code,
                        'lang' => $arrayLang,
                    );
                }
            }
        }

        return $return;
    }

    public static function syncProducts(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_product = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $return = array();

        $id_product = (!is_array($id_product) && $id_product > 0) ? array($id_product) : $id_product;

        $query = 'SELECT ' . ($justCount ? 'COUNT(b.`id_product`) AS `total`' :
        'b.`id_product`'.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
            ', product_shop.id_shop' : ', NULL AS id_shop')) . '
        FROM `' . _DB_PREFIX_ . 'product` b
        ' . (self::isMultiShop($id_shop) ?
            Shop::addSqlAssociation('product', 'b') :
            _PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
            'INNER JOIN `' . _DB_PREFIX_ . 'product_shop` product_shop ON (product_shop.id_product = b.id_product)' :
            '') . '
        WHERE '.($id_product && is_array($id_product) ?
                'b.`id_product` IN (' . implode(',', array_map('intval', $id_product)) . ')' : '1') . '
        ' . ($lastUpdate ? ' AND b.`date_upd` >= "' . pSQL($lastUpdate) . '"' : '') . '
        ' . (self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '') . '
        ' . ($justCount ? '' : 'ORDER BY b.`date_upd`') . '
        ' . ($limit && ! $justCount ? ' LIMIT ' . (int) $start . ', ' . (int) $limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if ($justCount) {
            return array(
                'count' => $results[0]['total']
            );
        }

        if ($results && is_array($results) && count($results)) {
            foreach ($results as $row) {
                $return = array_merge($return, self::getProduct(
                    $row['id_product'],
                    false,
                    null,
                    $row['id_shop']
                ));
            }
        }

        return $return;
    }

    public static function getProduct(
        $id_product,
        $id_lang = false,
        $id_product_attribute = null,
        $id_shop = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $context = self::getContext($id_shop);
        $image_size = self::getOptimumImageSize();

        $sql = '
        SELECT b.`available_for_order`, '.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE')
                ? 'product_shop.`active`' :
                'b.`active`').
            ', b.`date_add`, b.`date_upd`, b.`quantity`, b.`reference`, b.`id_manufacturer`,
            b.`id_product`, c.`id_lang`, c.`name`, c.`description_short`, c.`description`,
             MAX(d.`id_image`) `id_image`, e.`legend`, c.`link_rewrite` AS product_link_rewrite, b.`ean13`
        FROM `' . _DB_PREFIX_ . 'product` b
        ' . (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlAssociation('product', 'b') :
                '') .
        'LEFT JOIN `' . _DB_PREFIX_ . 'product_lang` c ON (b.`id_product` = c.`id_product`' .
            (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlRestrictionOnLang('c') : '') .
            ($id_lang ? 'AND c.id_lang = ' . (int) $id_lang : '') . ')
        ' . (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ? '
        LEFT JOIN `' . _DB_PREFIX_ . 'image` p ON (p.`id_product` = b.`id_product`)
        LEFT JOIN `'._DB_PREFIX_.'image_shop` d
        ON (p.`id_image` = d.`id_image` AND d.cover=1 AND d.id_shop='.(int)$context->shop->id.')' :
            'LEFT JOIN `' . _DB_PREFIX_ . 'image` d ON (d.`id_product` = b.`id_product` AND d.`cover` = 1)') . '
        LEFT JOIN `' . _DB_PREFIX_ . 'image_lang` e ON (d.`id_image` = e.`id_image` AND e.`id_lang` = c.`id_lang`)
        WHERE b.`id_product` = ' . (int) $id_product.($id_lang ? '' : ' GROUP BY c.`id_lang` 
        ORDER BY '.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                    'product_shop.active' : 'b.active'));

        $resultProduct = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($sql);

        $returnProduct = array();
        if ($resultProduct && is_array($resultProduct) && count($resultProduct)) {
            $currency = Currency::getDefaultCurrency();
            $link = $context->link;
            $product_categories = (method_exists('Product', 'getProductCategories') ?
                Product::getProductCategories($id_product) :
                Product::getIndexedCategories($id_product));

            foreach ($resultProduct as $row) {
                $active = (boolean) ($row['available_for_order'] && $row['active']);
                $lang_status = Language::getLanguage($row['id_lang']);
                $lang = Language::getIsoById($row['id_lang']);

                if (!$lang) {
                    continue;
                }

                if (is_array($lang_status) && array_key_exists('active', $lang_status) && $lang_status['active'] == 0) {
                    continue;
                }

                $row['id_image'] = Product::defineProductImage($row, $row['id_lang']);
                // Get price
                $price = Product::getPriceStatic(
                    $row['id_product'],
                    ! (bool) Group::getDefaultPriceDisplayMethod(),
                    false,
                    6,
                    null,
                    false,
                    false
                );
                // Get price discount
                $price_discount = Product::getPriceStatic(
                    $row['id_product'],
                    ! (bool) Group::getDefaultPriceDisplayMethod(),
                    false,
                    6,
                    null,
                    false,
                    true
                );

                $cover = Image::getCover($row['id_product']);

                $image_url = "";
                if (!is_null($link)) {
                    $image_url = $link->getImageLink(
                        $row['product_link_rewrite'],
                        $row['id_product'].'-'.$cover['id_image'],
                        $image_size
                    );

                    if (is_null($image_url) || !$image_url || $image_url == '') {
                        // Get image URL
                        $image_url = $link->getImageLink(
                            $row['product_link_rewrite'],
                            $row['id_product'].'-'.$row['id_image'],
                            $image_size
                        );
                    }
                }

                if (!preg_match('#^http#i', $image_url)) {
                    $image_url = _PS_BASE_URL_ . $image_url;
                }

                if (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE')) {
                    // Get product URL
                    $product_url = $link->getProductLink(
                        (int) $row['id_product'],
                        null,
                        null,
                        null,
                        (int) $row['id_lang'],
                        $context->shop->id
                    );
                    $image_url = self::getRealMultiShopUrl($image_url, $context->shop->id);
                } else {
                    $product_url = (!is_null($link)) ? $link->getProductLink(
                        (int) $row['id_product'],
                        null,
                        null,
                        null,
                        (int) $row['id_lang']
                    ) : '';
                }

                $combination = self::getAllProductCombination(
                    $row['id_product'],
                    $row['id_lang'],
                    $image_size,
                    $id_product_attribute,
                    $context->shop->id,
                    $image_url
                );

                if (isset($id_product_attribute)) {
                    if ($active && Configuration::get('SPM_excludeOutOfStockProducts')) {
                        // Dans le cas où le produit est actif, on va vérifier le stock restant pour cette déclinaison. Si pas de stock, on la désactive
                        $stockForAttribute = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
											SELECT b.`quantity`
											FROM `' . _DB_PREFIX_ . 'stock_available` b
											' . (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ? Shop::addSqlAssociation('product', 'b') : '') . '
											WHERE b.`id_product` = ' . (int) $id_product . ' AND b.`id_product_attribute` = ' . (int) $id_product_attribute . ' AND b.id_shop='.(int)$context->shop->id);

                        if (is_array($stockForAttribute) && count($stockForAttribute) > 0 && array_key_exists('quantity', $stockForAttribute[0]) && $stockForAttribute[0]['quantity'] == 0) {
                            $active = false;
                        }
                    }

                    $returnProduct[] = array(
                        'shop_id_shop' => (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                            $context->shop->id :
                            false),
                        'id_product' => $row['id_product'],
                        'reference' => $combination['reference'],
                        'ean13' => $combination['ean13'],
                        'lang' => $lang,
                        'name' => $row['name'],
                        'combination_name' => $combination['combination_name'],
                        'description_short' => trim(strip_tags(nl2br($row['description_short']))),
                        'description' => trim(strip_tags(nl2br($row['description']))),
                        'product_link' => isset($combination['product_link']) && $combination['product_link'] !== '' ?
                            $combination['product_link'] :
                            $product_url,
                        'image_link' => isset($combination['image_link']) && $combination['image_link'] !== '' ?
                            $combination['image_link'] :
                            $image_url,
                        'images' => self::getProductImages(
                            $context,
                            $row['product_link_rewrite'],
                            $row['id_lang'],
                            $row['id_product'],
                            $id_product_attribute
                        ),
                        'id_combination' => $combination['id_combination'],
                        'id_categories' => $product_categories,
                        'id_manufacturer' => $row['id_manufacturer'],
                        'currency' => $currency->iso_code,
                        'price' => (isset($combination['price']) && $combination['price'] !== '') ?
                            $combination['price'] :
                            $price,
                        'price_discount' => isset($combination['price_discount']) &&
                        $combination['price_discount'] !== '' ?
                            $combination['price_discount'] :
                            $price_discount,
                        'quantity_remaining' => isset($combination['quantity']) &&
                        $combination['quantity'] !== '' ?
                            $combination['quantity'] :
                            $row['quantity'],
                        'date_creation' => $row['date_add'],
                        'date_update' => $row['date_upd'],
                        'active' => (int)$active,
                        'custom' => array()
                    );
                } else {
                    if ($active && Configuration::get('SPM_excludeOutOfStockProducts')) {
                        // Dans le cas où le produit est actif, on va vérifier le stock restant pour ce produit. Si pas de stock, on le désactive
                        $stockForAttribute = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
											SELECT b.`quantity`
											FROM `' . _DB_PREFIX_ . 'stock_available` b
											WHERE b.`id_product` = ' . (int) $id_product . ' AND b.`id_product_attribute` = 0 AND b.id_shop='.(int)$context->shop->id);

                        if (is_array($stockForAttribute) && count($stockForAttribute) > 0 && array_key_exists('quantity', $stockForAttribute[0]) && $stockForAttribute[0]['quantity'] == 0) {
                            $active = false;
                        }
                    }

                    // Si le prix du produit est à 0, on va prendre le prix de la déclinaison par défaut
                    if (is_array($combination) && count($combination)) {
                        foreach ($combination as $comb) {
                            if ($comb['default']) {
                                $price = $comb['price'];
                                $price_discount = $comb['price_discount'];
                                break;
                            }
                        }
                    }

                    $returnProduct[] = array(
                        'shop_id_shop' => (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                            $context->shop->id :
                            false),
                        'id_product' => $row['id_product'],
                        'reference' => $row['reference'],
                        'ean13' => $row['ean13'],
                        'lang' => $lang,
                        'name' => $row['name'],
                        'description_short' => trim(strip_tags(nl2br($row['description_short']))),
                        'description' => trim(strip_tags(nl2br($row['description']))),
                        'product_link' => $product_url,
                        'image_link' => $image_url,
                        'images' => self::getProductImages(
                            $context,
                            $row['product_link_rewrite'],
                            $row['id_lang'],
                            $row['id_product']
                        ),
                        'combinations' => $combination,
                        'id_categories' => $product_categories,
                        'id_manufacturer' => $row['id_manufacturer'],
                        'currency' => $currency->iso_code,
                        'price' => $price,
                        'price_discount' => $price_discount,
                        'quantity_remaining' => $row['quantity'],
                        'date_creation' => $row['date_add'],
                        'date_update' => $row['date_upd'],
                        'active' => (int)$active,
                        'custom' => array()
                    );
                }
            }
        }

        return $id_lang ? reset($returnProduct) : $returnProduct;
    }

    public static function getOptimumImageSize()
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $productImageSize = Configuration::get('SPM_productImageSize');
        if ($productImageSize) {
            return $productImageSize;
        }
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow('
        SELECT `name`
        FROM `'._DB_PREFIX_.'image_type`
        WHERE `width` >= 200 AND height >= 200 AND `products` = 1
        ORDER BY `width` ASC');

        return array_key_exists('name', $result) ? $result['name'] : false;
    }

    public static function getRealMultiShopUrl(
        $url,
        $id_shop
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        if (_PS_VERSION_ < 1.5 || (int)$id_shop === (int)Configuration::get('PS_SHOP_DEFAULT')) {
            return $url;
        }

        if (Configuration::get('PS_SSL_ENABLED')) {
            $main_domain_default = Db::getInstance()->getValue('
            SELECT CONCAT(domain_ssl,physical_uri,virtual_uri)
            FROM '._DB_PREFIX_.'shop_url
            WHERE main=1 AND id_shop = '.(int)Configuration::get('PS_SHOP_DEFAULT'));

            $main_domain = Db::getInstance()->getValue('
            SELECT CONCAT(domain_ssl,physical_uri,virtual_uri)
            FROM '._DB_PREFIX_.'shop_url
            WHERE main=1 AND id_shop = '.(int)$id_shop);
        } else {
            $main_domain_default = Db::getInstance()->getValue('
            SELECT CONCAT(domain,physical_uri,virtual_uri)
            FROM '._DB_PREFIX_.'shop_url
            WHERE main=1 AND id_shop = '.(int)Configuration::get('PS_SHOP_DEFAULT'));

            $main_domain = Db::getInstance()->getValue('
            SELECT CONCAT(domain,physical_uri,virtual_uri)
            FROM '._DB_PREFIX_.'shop_url
            WHERE main=1 AND id_shop = '.(int)$id_shop);
        }

        if (!preg_match('#'.preg_quote($main_domain_default).'#', $url)) {
            return $url;
        }

        return preg_replace('#'.preg_quote($main_domain_default).'#', $main_domain, $url);
    }

    public static function getAllProductCombination(
        $id_product,
        $id_lang,
        $image_size,
        $id_product_attribute = null,
        $id_shop = false,
        $img_parent_product = ''
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        if (is_null($id_lang)) {
            $id_lang = false;
        }

        $context = self::getContext($id_shop);
        $link = $context->link;
        $returnCombination = array();

        $resultProduct = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
        SELECT f.`id_product_attribute`, f.`default_on`, f.`quantity`, c.`name`, 
               MAX(d.`id_image`) AS product_img_default, pai.`id_image`, e.`legend`,
               c.`link_rewrite` AS product_link_rewrite, f.`ean13`
        FROM `'._DB_PREFIX_.'product_attribute` f
        '.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlAssociation('product_attribute', 'f') :
                '').'
        LEFT JOIN `'._DB_PREFIX_.'product_lang` c ON (f.`id_product` = c.`id_product`'.
            (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlRestrictionOnLang('c') :
                '').')
        LEFT JOIN `'._DB_PREFIX_.'product_attribute_image` pai ON 
        (pai.`id_product_attribute` = f.`id_product_attribute`)
        '.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                'LEFT JOIN `'._DB_PREFIX_.'image` p ON (p.`id_product` = f.`id_product`)
                LEFT JOIN `'._DB_PREFIX_.'image_shop` d
            ON (d.`id_image` = p.`id_image` AND d.cover=1 AND d.id_shop='.(int)$context->shop->id.')' :
            'LEFT JOIN `'._DB_PREFIX_.'image` d ON (d.`id_product` = f.`id_product` AND d.`cover` = 1)').'
        LEFT JOIN `'._DB_PREFIX_.'image_lang` e ON (e.`id_image` = pai.`id_image` AND e.`id_lang` = c.`id_lang`)
        WHERE f.`id_product` = '.(int)$id_product.'
        AND c.`id_lang` = '.(int)$id_lang.((!is_null($id_product_attribute) && $id_product_attribute) ?
                ' AND f.`id_product_attribute` = '.(int)$id_product_attribute :
                '').' GROUP BY f.`id_product_attribute`');

        if ($resultProduct && is_array($resultProduct) && count($resultProduct)) {
            foreach ($resultProduct as $row) {
                $combinationImage = false;
                $combination_name = '';
                $combination_values = array();
                $combinations = self::getProductAttributeCombinationsById(
                    $id_product,
                    $row['id_product_attribute'],
                    $id_lang
                );

                if (method_exists('Product', 'getCombinationImageById')) {
                    $combinationImage = Product::getCombinationImageById($row['id_product_attribute'], $id_lang);
                }

                $combinationImage['id_product'] = $id_product;

                if (array_key_exists('id_image', $row)) {
                    $combinationImage['id_image'] = $row['id_image'];
                } elseif (array_key_exists('product_img_default', $row)) {
                    $combinationImage['id_image'] = $row['product_img_default'];
                }

                if ($combinations && count($combinations)) {
                    // Generate combination name

                    foreach ($combinations as $row2) {
                        $combination_name .= $row2['public_group_name'].' '.$row2['attribute_name'].' ';
                        $combination_values[] = array(
                            'name' => $row2['public_group_name'],
                            'value' => $row2['attribute_name'],
                        );
                        if ($row2['reference']) {
                            $row['reference'] = $row2['reference'];
                        }
                        $row['quantity'] = $row2['quantity'];
                    }
                    $combination_name = trim($combination_name);
                }

                if ($combinationImage && count($combinationImage)) {
                    $row['id_image'] = Product::defineProductImage($combinationImage, $id_lang);
                }

                // Get price
                $price = Product::getPriceStatic(
                    $id_product,
                    !(bool)Group::getDefaultPriceDisplayMethod(),
                    $row['id_product_attribute'],
                    6,
                    null,
                    false,
                    false
                );

                // Get price discount
                $price_discount = Product::getPriceStatic(
                    $id_product,
                    !(bool)Group::getDefaultPriceDisplayMethod(),
                    $row['id_product_attribute'],
                    6,
                    null,
                    false,
                    true
                );

                // Get image URL
                $image_url = (!is_null($link)) ? $link->getImageLink(
                    (int)$row['product_link_rewrite'],
                    $row['id_image'],
                    $image_size
                ) : '';
                if (!preg_match('#^http#i', $image_url)) {
                    $image_url = _PS_BASE_URL_.$image_url;
                }

                if (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE')) {
                    // Get product URL
                    $product_url = $link->getProductLink(
                        (int)$id_product,
                        null,
                        null,
                        null,
                        (int)$id_lang,
                        $context->shop->id
                    );
                    $image_url = self::getRealMultiShopUrl($image_url, $context->shop->id);
                } else {
                    $product_url = (!is_null($link)) ? $link->getProductLink(
                        (int)$id_product,
                        null,
                        null,
                        null,
                        (int)$id_lang
                    ) : '';
                }

                $all_images = self::getProductImages(
                    $context,
                    $row['product_link_rewrite'],
                    $id_lang,
                    $id_product,
                    $row['id_product_attribute']
                );

                if (count($all_images) == 0 && $img_parent_product != '') {
                    /* Si la déclinaison ne posséde aucunes images, on lui applique l'image du parent */
                    $image_url = $img_parent_product;
                }

                if (count($all_images) > 0) {
                    foreach ($all_images as $imageDec) {
                        if (is_array($imageDec) && array_key_exists('default', $imageDec)
                        && $imageDec['default'] && array_key_exists('url', $imageDec) && $imageDec['url'] !== '') {
                            $image_url = $imageDec['url'];
                            break;
                        }
                    }
                }

                $returnCombination[] = array(
                    'reference' => (array_key_exists('reference', $row) && $row['reference'] ?
                        $row['reference'] :
                        ''),
                    'ean13' => $row['ean13'],
                    'product_link' => (!is_null($product_url) && $product_url ? $product_url : ''),
                    'image_link' => $image_url,
                    'images' => $all_images,
                    'combination_name' => $combination_name,
                    'id_combination' => $row['id_product_attribute'],
                    'values' => $combination_values,
                    'price' => $price,
                    'price_discount' => $price_discount,
                    'quantity_remaining' => $row['quantity'],
                    'default' => (int)$row['default_on'],
                );
            }
        }

        return (!is_null($id_product_attribute) && $id_product_attribute) ?
            reset($returnCombination) :
            $returnCombination;
    }


    public static function getProductAttributeCombinationsById(
        $id_product,
        $id_product_attribute,
        $id_lang
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        if ((_PS_VERSION_ >= 1.5) && !Combination::isFeatureActive()) {
            return array();
        }

        $sql = 'SELECT pa.*, a.`id_attribute`, ag.`id_attribute_group`, ag.`is_color_group`,
        agl.`name` AS group_name, agl.`public_name` AS public_group_name, al.`name` AS attribute_name
        '.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                ',product_attribute_shop.*' :
                '').'
        FROM `'._DB_PREFIX_.'product_attribute` pa
        '.(_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlAssociation('product_attribute', 'pa') :
                '').'
        LEFT JOIN `'._DB_PREFIX_.'product_attribute_combination` pac ON 
            pac.`id_product_attribute` = pa.`id_product_attribute`
        LEFT JOIN `'._DB_PREFIX_.'attribute` a ON a.`id_attribute` = pac.`id_attribute`
        LEFT JOIN `'._DB_PREFIX_.'attribute_group` ag ON ag.`id_attribute_group` = a.`id_attribute_group`
        LEFT JOIN `'._DB_PREFIX_.'attribute_lang` al ON (a.`id_attribute` = al.`id_attribute` 
            AND al.`id_lang` = '.(int)$id_lang.')
        LEFT JOIN `'._DB_PREFIX_.'attribute_group_lang` agl ON 
            (ag.`id_attribute_group` = agl.`id_attribute_group` AND agl.`id_lang` = '.(int)$id_lang.')
        WHERE pa.`id_product` = '.(int)$id_product.'
        AND pa.`id_product_attribute` = '.(int)$id_product_attribute.'
        GROUP BY pa.`id_product_attribute`, ag.`id_attribute_group`
        ORDER BY pa.`id_product_attribute`';

        $res = Db::getInstance()->executeS($sql);

        // Get quantity of each variations
        foreach ($res as $key => $row) {
            if (_PS_VERSION_ < 1.5) {
                $quantity = Product::getQuantity(
                    $row['id_product'],
                    $row['id_product_attribute']
                );
            } else {
                $quantity = StockAvailable::getQuantityAvailableByProduct(
                    $row['id_product'],
                    $row['id_product_attribute']
                );
            }
            $res[$key]['quantity'] = $quantity;
        }

        return $res;
    }

    public static function getProductImages(
        $context,
        $product_link_rewrite,
        $id_lang,
        $id_product,
        $id_product_attribute = null
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $image_size = self::getOptimumImageSize();
        $link = $context->link;

        $images = Image::getImages($id_lang, $id_product, $id_product_attribute);

        $arrayImages = array();
        $hasDefault = false;
        if (is_array($images) && count($images) > 0) {
            foreach ($images as $image) {
                if ($image['cover'] == 1) {
                    $hasDefault = true;
                }

                $image_url = (!is_null($link)) ? $image_url = $link->getImageLink(
                    $product_link_rewrite,
                    $id_product.'-'.$image['id_image'],
                    $image_size
                ) : '';

                if (!preg_match('#^http#i', $image_url)) {
                    $image_url = _PS_BASE_URL_.$image_url;
                }

                if (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE')) {
                    $image_url = self::getRealMultiShopUrl($image_url, $context->shop->id);
                }

                array_push($arrayImages, array(
                    'id_image' => $image['id_image'],
                    'id_product' => $id_product,
                    'id_product_attribute' => $id_product_attribute,
                    'url' => $image_url,
                    'default' => $image['cover'],
                ));
            }

            if (!$hasDefault) {
                $arrayImages[0]['default'] = 1;
            }
        }

        return $arrayImages;
    }

    public static function syncProductsCategories(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_category = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        $id_category = (!is_array($id_category) && $id_category > 0) ? array($id_category) : $id_category;

        $query = 'SELECT '.($justCount ? 'COUNT(b.`id_category`) AS `total`' : 'b.`id_category`'.
                (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                    ', category_shop.id_shop AS id_shop' :
                    (_PS_VERSION_ >= 1.5 ?
                        ', b.id_shop_default AS id_shop' :
                        ', 0 AS id_shop'))).'
        FROM `'._DB_PREFIX_.'category` b
        '.(self::isMultiShop($id_shop) ?
                Shop::addSqlAssociation('category', 'b') :
                _PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                    'INNER JOIN `'._DB_PREFIX_.'category_shop` category_shop ON 
                        (category_shop.id_category = b.id_category)'
                    : '').'
        WHERE '.($id_category && is_array($id_category) ?
                'b.`id_category` IN ('.implode(',', array_map('intval', $id_category)).')' :
                '1').'
        '.($lastUpdate ? ' AND b.`date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
        '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '').'
        '.($justCount ? '' : 'ORDER BY b.`date_upd`').'
        '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
        if ($justCount) {
            return array(
                'count' => $results[0]['total'],
            );
        }
        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return = array_merge($return, self::getProductCategory($row['id_category'], $row['id_shop']));
            }
        }

        return $return;
    }

    public static function getProductCategory(
        $id_category,
        $idShop = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $context = self::getContext($idShop);

        $resultCategory = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
        SELECT a.`date_add`, a.`date_upd`, a.`id_parent`, a.`id_category`, a.`active`,
               b.`name`, b.`description`, b.`link_rewrite`, b.`id_lang`
        FROM `'._DB_PREFIX_.'category` a '.(_PS_VERSION_ >= 1.5 &&
            Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlAssociation('category', 'a') :
                '').'
        LEFT JOIN `'._DB_PREFIX_.'category_lang` b ON (a.`id_category` = b.`id_category`'.
            (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                Shop::addSqlRestrictionOnLang('b') :
                '').')
        WHERE a.`id_category` = '.(int)$id_category.' ORDER BY a.active');

        $return = array();
        if ($resultCategory && is_array($resultCategory) && sizeof($resultCategory)) {
            $link = $context->link;
            foreach ($resultCategory as $row) {
                $lang_status = Language::getLanguage($row['id_lang']);
                $lang = Language::getIsoById($row['id_lang']);

                if (!$lang) {
                    continue;
                }

                if (is_array($lang_status) && array_key_exists('active', $lang_status) && $lang_status['active'] == 0) {
                    continue;
                }

                // Get product URL
                if (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE')) {
                    $category_url = $link->getCategoryLink(
                        (int)$row['id_category'],
                        null,
                        (int)$row['id_lang'],
                        null,
                        $context->shop->id
                    );
                } else {
                    $category_url = (!is_null($link)) ? $link->getCategoryLink(
                        (int)$row['id_category'],
                        null,
                        (int)$row['id_lang']
                    ) : '';
                }

                $return[] = array(
                    'shop_id_shop' => (_PS_VERSION_ >= 1.5 && Configuration::get('PS_MULTISHOP_FEATURE_ACTIVE') ?
                        $context->shop->id :
                        false),
                    'id_category' => $row['id_category'],
                    'id_parent_category' => $row['id_parent'],
                    'lang' => $lang,
                    'name' => $row['name'],
                    'description' => trim(strip_tags(nl2br($row['description']))),
                    'link' => $category_url,
                    'date_creation' => $row['date_add'],
                    'date_update' => $row['date_upd'],
                    'active' => $row['active'],
                );
            }
        }

        return $return;
    }

    public static function syncOrders(
        $id_shop,
        $start,
        $limit,
        $lastUpdate,
        $id_order = false,
        $justCount = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $return = array();

        if ($lastUpdate) {
            $date = new DateTime($lastUpdate);
            $date->sub(new DateInterval('P12M'));
            $dateAddLimit = $date->format('Y-m-d H:i:s');
        }

        $id_order = (!is_array($id_order) && $id_order > 0) ? array($id_order) : $id_order;

        $query = 'SELECT '.($justCount ? 'COUNT(`id_order`) AS `total`' : '`id_order`').'
        FROM `'._DB_PREFIX_.'orders`
        WHERE '.($id_order && is_array($id_order) ? '`id_order` IN ('.implode(',', array_map('intval', $id_order)).')' : '1').'
        '.($lastUpdate ? ' AND `date_upd` >= "'.pSQL($lastUpdate).'"' : '').'
        '.(!empty($dateAddLimit) && !is_null($dateAddLimit) && $dateAddLimit ?
                ' AND `date_add` >= "'.pSQL($dateAddLimit).'"' :
                '').'
        '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '').'
        '.($justCount ? '' : 'ORDER BY `date_upd`').'
        '.($limit && !$justCount ? ' LIMIT '.(int)$start.', '.(int)$limit : '');

        $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if ($justCount) {
            return array(
                'count' => $results[0]['total'],
            );
        }

        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                $return[] = self::getOrder($row['id_order']);
            }
        }

        return $return;
    }

    public static function getOrder(
        $id_order,
        $orderState = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        if (is_object($id_order)) {
            $order = $id_order;
        } else {
            $order = new Order($id_order);
        }

        $currency = new Currency($order->id_currency);
        $cart = new Cart($order->id_cart);

        if (_PS_VERSION_ < '1.5.0.1') {
            $total_paid = $order->total_paid_real;
            $shipping_without_tax = round(
                $order->total_shipping / (1 + $order->carrier_tax_rate / 100),
                2
            );
        } else {
            $total_paid = $order->total_paid_tax_incl;
            $shipping_without_tax = $order->total_shipping_tax_excl;
        }

        $total_shipping = $order->total_shipping;

        if (!$total_paid && property_exists($order, 'total_paid') && $order->total_paid) {
            $total_paid = $order->total_paid;
        }

        $voucher_used = self::getOrderVoucher($order);
        $voucher_amount = $order->total_discounts;

        $amount_without_tax = $order->total_products;

        $return = array(
            'shop_id_shop' => ((property_exists($order, 'id_shop') &&
                $order->id_shop &&
                self::isMultiShop($order->id_shop)) ? $order->id_shop : null),
            'order_is_confirm' => ($orderState && is_object($orderState) ?
                (int)($orderState->logable) :
                $order->valid),
            'order_reference' => property_exists($order, 'reference') && $order->reference ?
                $order->reference :
                $order->id,
            'id_cart' => $order->id_cart,
            'id_carrier' => $order->id_carrier,
            'id_address_delivery' => $order->id_address_delivery,
            'id_address_invoice' => $order->id_address_invoice,
            'id_status' => ($orderState && is_object($orderState) ?
                $orderState->id :
                $order->getCurrentState()),
            'date_cart' => ($cart->date_upd !== null && $cart->date_upd ?
                $cart->date_upd :
                $order->date_add),
            'id_order' => $order->id,
            'lang' => Language::getIsoById($order->id_lang),
            'amount' => $total_paid,
            'amount_without_tax' => $amount_without_tax,
            'shipping' => $total_shipping,
            'shipping_without_tax' => $shipping_without_tax,
            'tax_rate' => $currency->conversion_rate,
            'currency_rate' => $currency->conversion_rate,
            'currency' => $currency->iso_code,
            'date_order' => $order->date_add,
            'date_update' => $order->date_upd,
            'voucher_used' => $voucher_used,
            'voucher_amount' => $voucher_amount,
            'products' => self::getSyncOrderProducts($order),
            'customer' => self::getUser($order->id_customer),
            'shipping_number' => $order->shipping_number,
            'custom' => array(),
        );

        return $return;
    }

    public static function getOrderVoucher($order)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $voucherUsed = array();
        if (_PS_VERSION_ < '1.5.0.1') {
            $vouchersOrder = $order->getDiscounts();
        } else {
            $vouchersOrder = $order->getCartRules();
        }
        if ($vouchersOrder && is_array($vouchersOrder) && sizeof($vouchersOrder)) {
            foreach ($vouchersOrder as $voucher) {
                if (_PS_VERSION_ < 1.5 && array_key_exists('id_discount', $voucher)) {
                    $cartRule = new Discount($voucher['id_discount']);
                    $voucherUsed[] = $cartRule->name;
                } elseif (array_key_exists('id_cart_rule', $voucher)) {
                    $cartRule = new CartRule($voucher['id_cart_rule']);
                    $voucherUsed[] = $cartRule->code;
                } else {
                    $voucherUsed[] = $voucher['name'];
                }
            }
        }

        return $voucherUsed;
    }

    public static function getSyncOrderProducts($order)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $context = self::getContext();

        if (!defined('_PS_BASE_URL_')) {
            require_once(dirname(__FILE__).'/../../../init.php');
        }

        if ($context->cart === null ||
            !is_object($context->cart) ||
            !property_exists($context->cart, 'id') ||
            is_null($context->cart->id) ||
            empty($context->cart->id) ||
            !$context->cart->id) {
            $context->cart = new Cart($order->id_cart);
        }

        $orderProducts = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('
        SELECT b.`id_product`, b.`id_manufacturer`, a.`unit_price_tax_incl`,
               a.`unit_price_tax_excl`, a.`product_quantity`, a.`product_attribute_id`
        FROM `'._DB_PREFIX_.'order_detail` a
        JOIN `'._DB_PREFIX_.'product` b ON (b.`id_product` = a.`product_id`)
        WHERE a.`id_order` = '.(int)$order->id.'
        GROUP BY `id_order_detail`');

        $returnProducts = array();
        if ($orderProducts && is_array($orderProducts) && count($orderProducts)) {
            foreach ($orderProducts as $row) {
                $product_categories = (method_exists('Product', 'getProductCategories') ?
                    Product::getProductCategories($row['id_product']) :
                    Product::getIndexedCategories($row['id_product']));

                $returnProducts[] = array(
                    'id_product' => $row['id_product'],
                    'id_combination' => $row['product_attribute_id'],
                    'id_manufacturer' => $row['id_manufacturer'],
                    'price' => $row['unit_price_tax_incl'],
                    'price_without_tax' => $row['unit_price_tax_excl'],
                    'qty' => $row['product_quantity'],
                    'id_categories' => $product_categories,
                );
            }
        }

        return $returnProducts;
    }

    /**
     * ******************* **
     * END SYNC REQUESTS *
     * * ******************* *
     */

    public static function generateVouchers(
        $voucherInfos,
        $emails,
        $id_shop
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $vouchers = array();

        if ($voucherInfos && is_array($voucherInfos) && $emails && is_array($emails) && sizeof($emails)) {
            foreach ($emails as $email) {
                if (is_array($email) &&
                    array_key_exists('email', $email) &&
                    array_key_exists('description', $email)) {
                    $voucher = self::generateVoucher(
                        $email['email'],
                        $voucherInfos['type'],
                        ($voucherInfos['type'] == 'shipping' ? false : $voucherInfos['amount']),
                        ($voucherInfos['type'] == 'amount' ? $voucherInfos['amountCurrency'] : false),
                        $voucherInfos['minimumOrder'],
                        $voucherInfos['nbDayValidate'],
                        (array_key_exists('description', $email) && (boolean)$email['description'] ?
                            $email['description'] :
                            null),
                        $id_shop,
                        (array_key_exists('dynamicPrefix', $voucherInfos) ?
                            $voucherInfos['dynamicPrefix'] :
                            false),
                        (array_key_exists('duplicateCode', $voucherInfos) ?
                            $voucherInfos['duplicateCode'] :
                            false),
                        (array_key_exists('codeToGenerate', $voucherInfos) ?
                            $voucherInfos['codeToGenerate'] :
                            false)
                    );
                    $email = $email['email'];
                } else {
                    $voucher = self::generateVoucher(
                        $email,
                        $voucherInfos['type'],
                        ($voucherInfos['type'] == 'shipping' ? false : $voucherInfos['amount']),
                        ($voucherInfos['type'] == 'amount' ? $voucherInfos['amountCurrency'] : false),
                        $voucherInfos['minimumOrder'],
                        $voucherInfos['nbDayValidate'],
                        null,
                        $id_shop,
                        (array_key_exists('dynamicPrefix', $voucherInfos) ?
                            $voucherInfos['dynamicPrefix'] :
                            false),
                        (array_key_exists('duplicateCode', $voucherInfos) ?
                            $voucherInfos['duplicateCode'] :
                            false),
                        (array_key_exists('codeToGenerate', $voucherInfos) ?
                            $voucherInfos['codeToGenerate'] :
                            false)
                    );
                }
                if (!$voucher) {
                    continue;
                }
                $vouchers[$email] = $voucher;
            }
        }

        return $vouchers;
    }

    public static function generateVoucher(
        $email,
        $type,
        $amount,
        $amountCurrency,
        $minimumOrder,
        $nbDayValidate,
        $description,
        $id_shop,
        $dynamicPrefix,
        $duplicateCode,
        $forceCodeName = false
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $date = date('Y-m-d H:i:s');
        $voucher_date_limit = date(
            'Y-m-d 23:59:59',
            mktime(
                date("H"),
                date("i"),
                date("s"),
                date("m"),
                date("d") + $nbDayValidate,
                date("Y")
            )
        );

        if (!$forceCodeName) {
            if ($dynamicPrefix === false) {
                $coupon_code = 'SPM-'.Tools::strtoupper(Tools::passwdGen(4));
            } else {
                $coupon_code = 'SPM-'.$dynamicPrefix.'-'.Tools::strtoupper(Tools::passwdGen(4));
            }
        } else {
            $coupon_code = $forceCodeName;
        }


        /**
         * Should we clone an existing rule or create a new one.
         */
        if ($duplicateCode !== false) {
            if (_PS_VERSION_ < 1.5) {
                $id_cart_rule = Discount::discountExists($duplicateCode);
            } else {
                $id_cart_rule = CartRule::getIdByCode($duplicateCode);

                if (!$id_cart_rule || $id_cart_rule == '' || !count($id_cart_rule)) {
                    // On effectue directement une requête pour récupérer l'id du code
                    $id_cart_rule_db = Db::getInstance()->ExecuteS('SELECT `id_cart_rule` 
                    FROM `'._DB_PREFIX_.'cart_rule` WHERE `code` = \''.pSQL($duplicateCode).'\'');

                    if ($id_cart_rule_db &&
                        is_array($id_cart_rule_db) &&
                        count($id_cart_rule_db) &&
                        array_key_exists('id_cart_rule', $id_cart_rule_db[0])) {
                        $id_cart_rule = $id_cart_rule_db[0]['id_cart_rule'];
                    }
                }
            }

            if (!$id_cart_rule || $id_cart_rule == '' || !count($id_cart_rule)) {
                return false;
            }

            if (_PS_VERSION_ < 1.5) {
                $cartRule = new Discount($id_cart_rule['id_discount']);
            } else {
                $cartRule = new CartRule($id_cart_rule);
            }

            $cartRule2 = clone $cartRule;
            $cartRule2->id = null;

            if (_PS_VERSION_ < 1.5) {
                $cartRule2->name = $coupon_code;

                // Set description
                foreach (array_keys($cartRule2->description) as $k) {
                    $cartRule2->description[$k] = (!is_null($description) &&
                        !empty($description)) ? $description : $coupon_code;
                }

                // Set categories
                $categories = Discount::getCategories($cartRule->id);
                $cartRule2->categories = array();
                if (is_array($categories) && count($categories)) {
                    foreach ($categories as $k => $category) {
                        $cartRule2->categories[] = $category['id_category'];
                    }
                }

                // Minimum Order
                if ($minimumOrder) {
                    $cartRule2->minimal = (float)$minimumOrder;
                    if (property_exists('Discount', 'include_tax')) {
                        $cartRule2->include_tax = 1;
                    }
                } else {
                    $cartRule2->minimal = 0;
                }

                // Forces group and customer restriction to none.
                $cartRule2->id_group = 0;
                $cartRule2->id_customer = 0;
            } else {
                $cartRule2->code = $coupon_code;
                foreach (array_keys($cartRule2->name) as $k) {
                    $cartRule2->name[$k] = $coupon_code;
                }

                if ($minimumOrder) {
                    $cartRule->minimum_amount = (float)$minimumOrder;
                    $cartRule->minimum_amount_tax = 1;
                    $cartRule->minimum_amount_currency = Configuration::get('PS_CURRENCY_DEFAULT');
                } else {
                    $cartRule->minimum_amount = 0;
                }

                $cartRule2->group_restriction = 0;
                $cartRule2->id_customer = 0;
            }

            if ((int)Configuration::get('SPM_voucherNominative')) {
                $id_customer = self::getIdCustomersByEmail($email);
                if ($id_customer) {
                    $cartRule2->id_customer = (int)$id_customer;
                    $customer = new Customer((int) $id_customer);
                    $isGuest = property_exists($customer, 'is_guest') && $customer->is_guest;

                    $voucher_highlight = (int)Configuration::get('SPM_voucherHighlight');
                    if ($voucher_highlight && !$isGuest) {
                        if (_PS_VERSION_ >= 1.5) {
                            $cartRule2->highlight = $voucher_highlight;
                        } elseif (_PS_VERSION_ >= 1.4) {
                            $cartRule2->cart_display = $voucher_highlight;
                        }
                    }
                }
            }

            $cartRule2->date_to = $voucher_date_limit;
            $cartRule2->date_from = $date;
            $cartRule2->quantity = 1;
            $cartRule2->quantity_per_user = 1;
            $cartRule2->active = 1;

            if (_PS_VERSION_ < 1.5) {
                if (!$cartRule2->add(true, false, $cartRule2->categories)) {
                    return false;
                }
            } else {
                if (!$cartRule2->add()) {
                    return false;
                }
            }

            // Copy Cart Rule Conditions
            if (_PS_VERSION_ > 1.5) {
                Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_shop` (`id_cart_rule`, `id_shop`) 
                (SELECT '.(int)$cartRule2->id.', id_shop 
                FROM `'._DB_PREFIX_.'cart_rule_shop` WHERE `id_cart_rule` = '.(int)$cartRule->id.')');

                Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_carrier` 
                (`id_cart_rule`, `id_carrier`) 
                (SELECT '.(int)$cartRule2->id.', id_carrier 
                FROM `'._DB_PREFIX_.'cart_rule_carrier` WHERE `id_cart_rule` = '.(int)$cartRule->id.')');

                Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_country`
                 (`id_cart_rule`, `id_country`)
                 (SELECT '.(int)$cartRule2->id.', id_country 
                 FROM `'._DB_PREFIX_.'cart_rule_country` WHERE `id_cart_rule` = '.(int)$cartRule->id.')');

                Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_combination` 
                (`id_cart_rule_1`, `id_cart_rule_2`) 
                (SELECT '.(int)$cartRule2->id.', IF(id_cart_rule_1 != '.(int)$cartRule->id.', 
                id_cart_rule_1, id_cart_rule_2) FROM `'._DB_PREFIX_.'cart_rule_combination` 
                WHERE `id_cart_rule_1` = '.(int)$cartRule->id.' OR `id_cart_rule_2` = '.(int)$cartRule->id.')');

                // Copy products/category filters
                $products_rules_group_source = Db::getInstance()->ExecuteS('SELECT id_product_rule_group,quantity
                FROM `'._DB_PREFIX_.'cart_rule_product_rule_group` WHERE `id_cart_rule` = '.(int)$cartRule->id.' ');

                foreach ($products_rules_group_source as $product_rule_group_source) {
                    Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_product_rule_group`
                     (`id_cart_rule`, `quantity`) VALUES ('.(int)$cartRule2->id.','.
                        (int)$product_rule_group_source['quantity'].')');

                    $id_product_rule_group_destination = Db::getInstance()->Insert_ID();
                    $products_rules_source = Db::getInstance()->ExecuteS('SELECT id_product_rule,type 
                    FROM `'._DB_PREFIX_.'cart_rule_product_rule` WHERE `id_product_rule_group` = '.
                        (int)$product_rule_group_source['id_product_rule_group'].' ');

                    foreach ($products_rules_source as $product_rule_source) {
                        Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_product_rule`
                         (`id_product_rule_group`, `type`) VALUES ('.
                            (int)$id_product_rule_group_destination.',"'.pSQL($product_rule_source['type']).'")');

                        $id_product_rule_destination = Db::getInstance()->Insert_ID();
                        $products_rules_values_source = Db::getInstance()->ExecuteS('SELECT id_item 
                        FROM `'._DB_PREFIX_.'cart_rule_product_rule_value`
                        WHERE `id_product_rule` = '.(int)$product_rule_source['id_product_rule'].' ');

                        foreach ($products_rules_values_source as $product_rule_value_source) {
                            Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_product_rule_value`
                             (`id_product_rule`, `id_item`) VALUES ('.
                                (int)$id_product_rule_destination.','.(int)$product_rule_value_source['id_item'].')');
                        }
                    }
                }
            }
        } else {
            if (_PS_VERSION_ < 1.5) {
                $cartRule = new Discount();
                $cartRule->name = $coupon_code;
                if ($type == 'percent') {
                    $cartRule->id_discount_type = 1;
                    $cartRule->value = $amount;
                } elseif ($type == 'amount') {
                    $cartRule->id_discount_type = 2;
                    $cartRule->value = (float)$amount;
                    $id_currency_amount = false;
                    if ($amountCurrency) {
                        $id_currency_amount = Currency::getIdByIsoCode($amountCurrency);
                    }
                    if (!$id_currency_amount) {
                        $id_currency_amount = Configuration::get('PS_CURRENCY_DEFAULT');
                    }
                    $cartRule->id_currency = (int)$id_currency_amount;
                } elseif ($type == 'shipping') {
                    $cartRule->id_discount_type = 3;
                    $cartRule->value = 0;
                }
                $languages = Language::getLanguages(true);
                foreach ($languages as $language) {
                    $cartRule->description[(int)$language['id_lang']] = (!is_null($description) &&
                        !empty($description)) ?
                        $description :
                        $coupon_code;
                }
                if ($minimumOrder) {
                    $cartRule->minimal = (float)$minimumOrder;
                    if (property_exists('Discount', 'include_tax')) {
                        $cartRule->include_tax = 1;
                    }
                } else {
                    $cartRule->minimal = 0;
                }
                $cartRule->cumulable = (bool)Configuration::get('SPM_voucherCumulableVoucher');
                $cartRule->cumulable_reduction = (bool)Configuration::get('SPM_voucherCumulablePromo');
            } else {
                $cartRule = new CartRule();
                if ($type == 'percent') {
                    $cartRule->reduction_percent = (float)$amount;
                } elseif ($type == 'amount') {
                    $cartRule->reduction_amount = (float)$amount;
                    $cartRule->reduction_tax = 1;
                    $id_currency_amount = false;
                    if ($amountCurrency) {
                        $id_currency_amount = Currency::getIdByIsoCode($amountCurrency);
                    }
                    if (!$id_currency_amount) {
                        $id_currency_amount = Configuration::get('PS_CURRENCY_DEFAULT');
                    }
                    $cartRule->reduction_currency = (int)$id_currency_amount;
                } elseif ($type == 'shipping') {
                    $cartRule->free_shipping = 1;
                }
                $cartRule->code = $coupon_code;
                $languages = Language::getLanguages(true);
                foreach ($languages as $language) {
                    $cartRule->name[(int)$language['id_lang']] = (!is_null($description) &&
                        !empty($description)) ?
                        $description :
                        $coupon_code;
                }

                if ($minimumOrder) {
                    $cartRule->minimum_amount = (float)$minimumOrder;
                    $cartRule->minimum_amount_tax = 1;
                    $cartRule->minimum_amount_currency = Configuration::get('PS_CURRENCY_DEFAULT');
                } else {
                    $cartRule->minimum_amount = 0;
                }
                $cartRule->cart_rule_restriction = ((bool)Configuration::get('SPM_voucherCumulableVoucher') === false ?
                    1 :
                    0);
            }

            if ((int)Configuration::get('SPM_voucherNominative')) {
                $id_customer = self::getIdCustomersByEmail($email);
                if ($id_customer) {
                    $cartRule->id_customer = (int)$id_customer;
                    $customer = new Customer((int) $id_customer);
                    $isGuest = property_exists($customer, 'is_guest') && $customer->is_guest;

                    $voucher_highlight = (int)Configuration::get('SPM_voucherHighlight');
                    if ($voucher_highlight && !$isGuest) {
                        if (_PS_VERSION_ >= 1.5) {
                            $cartRule->highlight = $voucher_highlight;
                        } elseif (_PS_VERSION_ >= 1.4) {
                            $cartRule->cart_display = $voucher_highlight;
                        }
                    }
                }
            }

            $cartRule->date_to = $voucher_date_limit;
            $cartRule->date_from = $date;
            $cartRule->quantity = 1;
            $cartRule->quantity_per_user = 1;
            $cartRule->active = 1;
            if (self::isMultiShop($id_shop)) {
                $cartRule->shop_restriction = true;
            }
            if (!$cartRule->add()) {
                return false;
            }
            if (self::isMultiShop($id_shop)) {
                $cartRule->shop_restriction = true;
                Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.'cart_rule_shop` (`id_cart_rule`, `id_shop`) 
                VALUES ('.(int)$cartRule->id.','.(int)$id_shop.') 
                ON DUPLICATE KEY UPDATE `id_shop` = '.(int)$id_shop.'');
            }
        }

        return array(
            'voucher_number' => $coupon_code,
            'voucher_date_limit' => $voucher_date_limit,
        );
    }

    public static function getIdCustomersByEmail($email)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow('SELECT `id_customer` FROM `'._DB_PREFIX_.'customer`
        WHERE `email` = \''.pSQL($email).'\' AND `is_guest` = 0');

        return array_key_exists('id_customer', $result) ? $result['id_customer'] : false;
    }

    public static function getCustomerGroups($id_shop)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $return = array();
        if (self::isMultiShop($id_shop)) {
            $results = Group::getGroups((int)Configuration::get('PS_LANG_DEFAULT'), (int)$id_shop);
        } else {
            $results = Group::getGroups((int)Configuration::get('PS_LANG_DEFAULT'));
        }
        if ($results) {
            foreach ($results as $row) {
                $return[$row['id_group']] = $row['name'];
            }
        }

        return $return;
    }

    public static function getLangs($id_shop)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $return = array();
        if (self::isMultiShop($id_shop)) {
            $results = Language::getLanguages(true, $id_shop);
        } else {
            $results = Language::getLanguages(true);
        }

        if ($results) {
            foreach ($results as $row) {
                $return[$row['iso_code']] = $row['name'];
            }
        }

        return $return;
    }

    public static function deleteUnusedVouchers()
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $date = date('Y-m-d H:i:s');
        if (_PS_VERSION_ < 1.5) {
            $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT `id_discount` AS `id` 
            FROM `'._DB_PREFIX_.'discount` WHERE `name` LIKE "SPM-%" AND `date_to` <= DATE_SUB("'.pSQL($date).'",
             INTERVAL '.(Configuration::get('SPM_DELETE_VOUCHER_INTERVAL') === false ?
                    7 :
                    (int)Configuration::get('SPM_DELETE_VOUCHER_INTERVAL')).' DAY)');
        } else {
            $results = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS('SELECT `id_cart_rule` AS `id` 
            FROM `'._DB_PREFIX_.'cart_rule` WHERE `code` LIKE "SPM-%" AND `date_to` <= DATE_SUB("'.pSQL($date).'",
             INTERVAL '.(Configuration::get('SPM_DELETE_VOUCHER_INTERVAL') === false ?
                    7 :
                    (int)Configuration::get('SPM_DELETE_VOUCHER_INTERVAL')).' DAY)');
        }

        if ($results && is_array($results) && sizeof($results)) {
            foreach ($results as $row) {
                if (_PS_VERSION_ < 1.5) {
                    $cartRule = new Discount($row['id']);
                } else {
                    $cartRule = new CartRule($row['id']);
                }
                $cartRule->delete();
            }
        }

        return true;
    }

    public static function getCartVouchers()
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $context = self::getContext();
        $cart = $context->cart;
        $discounts = array();
        if (_PS_VERSION_ < '1.5.0.1') {
            foreach ((array)$cart->getDiscounts(true) as $discount) {
                $discounts[] = $discount['name'];
            }
        } else {
            foreach ((array)$cart->getCartRules() as $discount) {
                $discounts[] = $discount['name'];
            }
        }

        return $discounts;
    }

    public static function saveProduct($id_product)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('product');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveProduct($id_product);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();
                $shops = self::getShopsForObject('product', $id_product);

                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!is_null(self::getApiIdentification()) &&
                        self::getApiIdentification() != '' &&
                        !in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveProduct($id_product);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('product');
        }
    }

    public static function checkSyncDate($type)
    {
        if (isset($type) || true) {
            return true;
        }
/*        $pass = true;
        $lastCall = Configuration::get('SPM_SYNCLASTCALL_'.Tools::strtoupper($type));

        if (!is_null($lastCall) && $lastCall && $lastCall != '') {
            $datetime1 = new DateTime();
            $datetime2 = new DateTime($lastCall);
            $interval = ($datetime1->format('U') - $datetime2->format('U')) / 60;
            if ($interval < 1) {
                $pass = false;
            }
        }

        return $pass;*/
    }

    public static function isMultiShopContext($context)
    {
        if (property_exists($context, 'shop') &&
            property_exists($context->shop, 'id') &&
            $context->shop->id && self::isMultiShop($context->shop->id)) {
            return true;
        }

        return false;
    }

    /**
     * Récupère la liste des ID shops liés à l'objet passé en paramètre
     * @param $objectType
     * @param $objectId
     * @return array
     */
    public static function getShopsForObject(
        $objectType,
        $objectId
    ) {
        $shops = array();

        switch ($objectType) {
            case 'product':
                $query = 'SELECT DISTINCT p.`id_shop` FROM `'._DB_PREFIX_.'product_shop` p 
                WHERE p.`id_product` = '.(int)$objectId.' ORDER BY p.`id_shop`';
                break;

            case 'productCategory':
                $query = 'SELECT DISTINCT p.`id_shop` FROM `'._DB_PREFIX_.'category_shop` p 
                WHERE p.`id_category` = '.(int)$objectId.' ORDER BY p.`id_shop`';
                break;

            case 'customerGroup':
                $query = 'SELECT DISTINCT p.`id_shop` FROM `'._DB_PREFIX_.'group_shop` p 
                WHERE p.`id_group` = '.(int)$objectId.' ORDER BY p.`id_shop`';
                break;

            case 'customer':
                $query = 'SELECT DISTINCT p.`id_shop` FROM `'._DB_PREFIX_.'customer` p 
                WHERE p.`id_customer` = '.(int)$objectId.' ORDER BY p.`id_shop`';
                break;

            case 'order':
                $query = 'SELECT DISTINCT p.`id_shop` FROM `'._DB_PREFIX_.'orders` p 
                WHERE p.`id_order` = '.(int)$objectId.' ORDER BY p.`id_shop`';
                break;
        }
        $qShops = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        if ($qShops && is_array($qShops) && count($qShops) > 0) {
            foreach ($qShops as $shop) {
                array_push($shops, $shop['id_shop']);
            }
        }

        return $shops;
    }

    public static function getApiIdentification()
    {
        return Configuration::get('SPM_apiIdentification');
    }

    public static function setSyncDate($type)
    {
        Configuration::updateValue('SPM_SYNCLASTCALL_'.Tools::strtoupper($type), date('Y-m-d H:i:s'));
    }

    public static function setScriptUrl($new_script_url)
    {
        Configuration::updateValue('SPM_SCRIPT_URL', $new_script_url);
    }

    public static function saveProductCategory($id_category)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('productCategory');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveProductCategory($id_category);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('productCategory', $id_category);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveProductCategory($id_category);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('productCategory');
        }
    }

    public static function saveCustomerGroup($id_group)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('customerGroup');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveCustomerGroup($id_group);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('customerGroup', $id_group);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveCustomerGroup($id_group);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('customerGroup');
        }
    }

    public static function saveManufacturer($id_manufacturer)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('manufacturer');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveManufacturer($id_manufacturer);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('manufacturer', $id_manufacturer);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveManufacturer($id_manufacturer);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('manufacturer');
        }
    }

    public static function saveCarrier($id_carrier)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('carrier');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveCarrier($id_carrier);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('carrier', $id_carrier);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveCarrier($id_carrier);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('carrier');
        }
    }

    public static function saveVoucher($id_voucher)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('voucher');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveVoucher($id_voucher);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('voucher', $id_voucher);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveVoucher($id_voucher);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('voucher');
        }
    }

    public static function saveOrderStatus($id_status)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('orderStatus');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveOrderStatus($id_status);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('orderStatus', $id_status);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveOrderStatus($id_status);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('orderStatus');
        }
    }

    public static function saveCustomer($id_customer)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $pass = self::checkSyncDate('customer');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveCustomer($id_customer);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('customer', $id_customer);

                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveCustomer($id_customer);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('customer');
        }
    }

    public static function saveOrder($id_order)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        $pass = self::checkSyncDate('order');
        if ($pass) {
            include_once(dirname(__FILE__).'/bin/Notify.php');

            if (!self::isMultishopContext(self::getContext())) {
                ShopimindClientNotify::saveOrder($id_order);
            } else {
                // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
                $currentContext = clone self::getContext();

                $shops = self::getShopsForObject('order', $id_order);
                $keysDone = array();
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::saveOrder($id_order);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }

                // On remet le contexte initial
                self::getContext($currentContext->shop->id);
            }

            self::setSyncDate('order');
        }
    }

    public static function deleteProduct($id_product)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_product' => $id_product,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteProduct($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('product', $id_product);
            $keysDone = array();

            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteProduct($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteProduct($params);
            }


            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    public static function deleteProductCategory($id_category)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_category' => $id_category,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteProductCategory($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('productCategory', $id_category);
            $keysDone = array();


            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteProductCategory($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteProductCategory($params);
            }

            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    public static function deleteCustomerGroup($id_group)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_group' => $id_group,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteCustomerGroup($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('customerGroup', $id_group);
            $keysDone = array();


            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteCustomerGroup($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteCustomerGroup($params);
            }

            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    public static function deleteManufacturer($id_manufacturer)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_manufacturer' => $id_manufacturer,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteManufacturer($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('manufacturer', $id_manufacturer);
            $keysDone = array();


            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteManufacturer($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteManufacturer($params);
            }


            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    public static function deleteCarrier($id_carrier)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_carrier' => $id_carrier,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteCarrier($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('carrier', $id_carrier);
            $keysDone = array();

            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteCarrier($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteCarrier($params);
            }
            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    public static function deleteOrderStatus($id_status)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_status' => $id_status,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteOrderStatus($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('orderStatus', $id_status);
            $keysDone = array();


            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteOrderStatus($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteOrderStatus($params);
            }

            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    public static function deleteCustomer($id_customer)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }
        include_once(dirname(__FILE__).'/bin/Notify.php');
        $params = array(
            'id_customer' => $id_customer,
        );
        if (!count($params)) {
            return false;
        }

        if (!self::isMultishopContext(self::getContext())) {
            ShopimindClientNotify::deleteCustomer($params);
        } else {
            // Dans le cas d'un multishop, on va boucler sur les shops concernés par la mise à jour
            $currentContext = clone self::getContext();

            $shops = self::getShopsForObject('customer', $id_customer);
            $keysDone = array();

            if (is_array($shops) && count($shops) > 0) {
                foreach ($shops as $shop) {
                    // Changement de contexte
                    self::getContext($shop);

                    if (!in_array(self::getApiIdentification(), $keysDone)) {
                        // Synchronisation
                        ShopimindClientNotify::deleteCustomer($params);
                        array_push($keysDone, self::getApiIdentification());
                    }
                }
            } else {
                ShopimindClientNotify::deleteCustomer($params);
            }

            // On remet le contexte initial
            self::getContext($currentContext->shop->id);
        }
    }

    /**
     * Désinscription d'un customer de la newsletter
     * @param $id_shop
     * @param $id_customer
     * @param $email
     * @return bool|mixed
     */
    public static function unsubCustomer(
        $id_shop,
        $id_customer,
        $email
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $query = 'UPDATE `'._DB_PREFIX_.'customer` a
                  SET a.`optin` = 0, a.`newsletter` = 0
                  WHERE a.`id_customer` = "'.pSQL($id_customer).'"
                  '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '');
        Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        // Si la synchro des listes newsletter est activée, on désinscrit de la newsletter
        $isEnabled = Configuration::get('SPM_syncNL');
        if ($isEnabled) {
            // Récupération de la configuration
            $table = Configuration::get('SPM_syncNL_table');
            $fields = explode('|', Configuration::get('SPM_syncNL_table_fields'));

            $tableFields = $finalFields = array();
            if (is_array($fields) && count($fields) > 0) {
                foreach ($fields as $field) {
                    $field = explode('-', $field);
                    $tableFields[$field[1]] = $field[0];
                    array_push($finalFields, $field[1]);
                }

                if (count($finalFields) > 0 &&
                    array_key_exists('email', $tableFields) &&
                    array_key_exists('status', $tableFields)) {
                    $query = 'UPDATE `'.$table.'` a
                    SET `'.$tableFields['status'].'` = 0 WHERE 1
                    '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false, 'a') : '').'
                    AND `'.bqSQL($tableFields['email']).'` = "'.pSQL($email).'"';
                    Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);
                }
            }
        }

        return true;
    }

    /**
     * Inscription d'un customer de la newsletter
     * @param $id_shop
     * @param $id_customer
     * @param $email
     * @return bool|mixed
     */
    public static function subscribeCustomer(
        $id_shop,
        $id_customer
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $query = 'UPDATE `'._DB_PREFIX_.'customer` a
                  SET a.`optin` = 1, a.`newsletter` = 1
                  WHERE a.`id_customer` = "'.pSQL($id_customer).'"
                  '.(self::isMultiShop($id_shop) ? Shop::addSqlRestriction(false) : '');
        Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        return true;
    }

    /**
     * Création d'un nouveau customer sur la boutique
     * @param $id_shop
     * @param $customerData
     * @return bool|mixed
     */
    public static function createCustomer(
        $id_shop,
        $customerData
    ) {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        $context = self::getContext($id_shop);

        // Checked the user address in case he changed his email address
        if (Validate::isEmail($email = $customerData['email']) && !empty($email)) {
            if (Customer::customerExists($email)) {
                return array('success' => false, 'message' => 'account_exists');
            }
        }

        // Preparing customer
        $customer = new Customer();
        $lastnameAddress = $customerData['lastName'];
        $firstnameAddress = $customerData['firstName'];

        $customer->lastname = $lastnameAddress;
        $customer->firstname = $firstnameAddress;
        $pwd = $customerData['password'];
        $customer->passwd = md5(pSQL(_COOKIE_KEY_.$pwd));
        $customer->email = $email;

        if (!Tools::isSubmit('submitGuestAccount')) {
            $customer->firstname = Tools::ucwords($customer->firstname);

            $customer->birthday = $customerData['birthday'];
            if (!Validate::isBirthDate($customer->birthday)) {
                return false;
            }

            // New Guest customer
            $customer->is_guest = (Tools::isSubmit('is_new_customer') ? !Tools::getValue('is_new_customer', 1) : 0);
            $customer->active = 1;
            $customer->newsletter = $customerData['newsletter'] == 1 ? 1 : 0;

            if ($customer->add()) {
                Mail::Send(
                    $context->language->id,
                    'account',
                    Mail::l('Welcome!'),
                    array(
                        '{firstname}' => $customer->firstname,
                        '{lastname}' => $customer->lastname,
                        '{email}' => $customer->email,
                        '{passwd}' => $customerData['password'],
                    ),
                    $customer->email,
                    $customer->firstname.' '.$customer->lastname
                );

                Hook::exec('actionCustomerAccountAdd', array(
                    //'_POST' => $_POST,
                    'newCustomer' => $customer,
                ));
                return true;
            } else {
                return array('success' => false, 'message' => 'error_add');
            }
        }

        return array('success' => false, 'message' => 'error');
    }

    public static function getApiPassword()
    {
        return Configuration::get('SPM_apiPassword');
    }

    public static function getApiClientUrl()
    {
        $module = new ShopiMind();

        return $module->getApiClientUrl();
    }

    public static function getTableFields($tableName)
    {
        if (class_exists('ShopimindClientCallbackOverride', false) &&
            method_exists('ShopimindClientCallbackOverride', __FUNCTION__)) {
            return call_user_func_array(array('ShopimindClientCallbackOverride', __FUNCTION__,), func_get_args());
        }

        if (is_null($tableName) || !$tableName || $tableName == '') {
            return false;
        }

        $query = 'SHOW COLUMNS FROM `'.bqSQL($tableName).'`';
        $result = Db::getInstance(_PS_USE_SQL_SLAVE_)->executeS($query);

        $fields = array();
        if ($result && is_array($result) && count($result)) {
            foreach ($result as $row) {
                array_push($fields, $row['Field']);
            }
        }

        return json_encode(array('success' => true, 'fields' => $fields));
    }
}
